!This module solves variably saturated flow in 3D karst matrix.
    
include 'mkl_pardiso.f90'
    
MODULE m3D_CV_IGA_Matrix
use mGlobalData
use BASISFUNCTION
implicit none
!This module solves mixed-form based formulation of 3D Richards equation for varibaly saturated groundwater flow.
!Discretizated by Fup (Spline) finite volume method in space and implicit (backward) Euler in time. 
!Modified Picard iteration technique is applied for linearization, where moisture content is expressed with respect to &
!hydraulic head by expanding in a truncated Taylor series.
!Prepared for Fup1 and B2.
!Written by: Luka Malenica, 01/2018.
! The second part of module solves 3D ADE; (depending variable: concentration) 
! in 3-D matrix using Euler-Langrangian framework and control volume grid used for flow solution. 
! Advective step is solved by backward particle method so that for every control volume particles
! are tracked backward from the related Gauss/quadrature points using the current time step. Due to advection, 
! concentration is the same along the trajectory along the current time step. It means that concentration 
! is the same in Gauss/quadrature point (indeed end point) as well as in the starting point of trajectory.Finaly, 
! advective concentration is solved as linear problem of function approximation.
! Dispersive step with time term and sink-source (advective flux exchange with 1-D conduits) is solved using
! the Control-Volume IsoGeometric Analysis (CV-IGA) in the same way and grid as for flow solution:   
! Spatial discretization: CV-IGA; prepared for Fup1 or B2-spline.
! Temporal discretization: Backward Euler.    
! Written by: Hrvoje Gotovac, Marin Zelenika and Luka Malenica, 04/2024.
PRIVATE 

    integer (kind=4) nx,ny,nz   !Number of CVs in each direction
    integer (kind=4) nxTOT,nyTOT,nzTOT   !Total number of basis function in each direction
    integer (kind=4) nSegments,nExternal       !Number of characteristic segments and external of basis function
    integer (kind=4) NTOT3D
    integer (kind=4) NNZ
    integer (kind=4) ixPipe1,ixPipe2,iyPipe,izPipe
    integer (kind=4) kmax,kmaxCI1D,kmax3D,kmaxMV,iter
    integer (kind=4) SCF
    integer (kind=4) icntNNZ,IURjcnt 
    integer (kind=4), allocatable :: indx(:),indy(:),indz(:)
    integer (kind=4), allocatable :: CNL(:),CNR(:)
    integer (kind=4), allocatable :: IA(:),JA(:)
    integer (kind=4), allocatable :: SoilType(:,:,:)
    

    real (kind=8) dlx1,dlx2,dly1,dly2,dlz1,dlz2     !Domain boundaries
    real (kind=8) dx,dy,dz  !Grid distance in each direction
    real (kind=8) TStart,TCurr,TStep,TEnd
    !real (kind=8) conmat1,conmat2,conmat3
    real (kind=8) ResidPress
    !real (kind=8) VGM_n,VGM_m,VGM_alpha,SpecificStorage,Porosity
    real (kind=8) epsNonLin
    real (kind=8) thetaRE
    real (kind=8) ImplicitUnderRelax,ExplicitUnderRelax,epsUnderRelax
    !real (kind=8) FluxRelaxation3DTemp
    real (kind=8) SatCond(3),AnisotropZ(3),VGM_n(3,2),VGM_m(3,2),VGM_alpha(3,2),VGM_w(3,2),VGM_tau(3),SpecificStorage(3),Porosity(3),ResidualWaterContent(3),ResidualSaturation(3)
    real(kind=8) LPCoordinates(44,3)    !Lower Piezometers (OrderNumber,x:z) ; OrderNumber goes from A01-D11
    real(kind=8) UPCoordinates(20,3)     !Upper Piezometers (OrderNumber,x:z) ; OrderNumber goes from 1-4 (4 pressure sensors installed into packers)
        
 
    
    real (kind=8), allocatable :: xVertex(:),yVertex(:),zVertex(:)
    real (kind=8), allocatable :: xCenter(:),yCenter(:),zCenter(:)
    real (kind=8), allocatable :: xcp1(:),xcp2(:),ycp1(:),ycp2(:),zcp1(:),zcp2(:)
    real (kind=8), allocatable :: BFIV(:,:)   !1D basis function integral value for unit caracteristic interval 
    real (kind=8), allocatable :: DELX(:),DELY(:),DELZ(:)   !Array of characteristic interval length
    real (kind=8), allocatable :: CC_t0(:),CC_lastIter(:),CC_t1(:)  !Solution head coefficients for 3D matrix
    real (kind=8), allocatable :: CCC_t0(:),CCC_lastIter(:),CCC_t1(:)  !Solution concentration coefficients for 3D matrix
    !real (kind=8), allocatable :: RHS_t0(:,:,:)
    real (kind=8), allocatable :: ASIMQ(:,:),APARADISO(:)
    
    character (len=20) LinSol,problem,writeSoE,CheckResidual,FullIntegration,UnsatSoilModel(3)
    character (len=20) BCType(6),BCType_Transport(6)
    
    logical WritePiezometricResults/.false./,WriteInitCond
    !logical InitLog/.true./
    
PUBLIC:: InitializeRE3D,SolveRE3D,Hm3D,Vm3D,Conc_3D,SolveAdvection3D,SolveDispersion3D,ModifyConduitPosition,ResultsRE3D,ResultsTransport3D,UpdateCoeffRE3D

    CONTAINS

    
!___________________________________________________________!
!PUBLIC DATA

    subroutine InitializeRE3D
        
        !Read input data
        call InputData
        !Create input data file
        call CreateCalcInputParamFile
        !Prepare calculation parameters
        call CalculateParameters
        !Prepare basis function integral values
        call CalculateIntegrals
        !Define saturated conductivity field
        call DefineSoilType
        !!Write conductivity field into file
        !call WriteConductivityField
        !stop
        !Transform of initial condition functions
        call InitialConditionTransform
        !Write results in output files
        !call WriteResults('InitialConditions')
        call WriteResults('PrepareFile')
        !Set initial condition as current solution
        CC_t1=CC_LastIter
        !Write initial condition results in output files
        WriteInitCond=.true.
        call ResultsRE3D
        WriteInitCond=.false.
        !This variable avoids writting initial condition pressure value
        WritePiezometricResults=.true.
    
!!***************************************************!
!    call TestSoilProperties
!    stop
!!***************************************************!

    end subroutine
    
    subroutine SolveRE3D(TimeE)
    
        integer (kind=4) icnteps,CoorEps(3)
        real (kind=8) TimeE
        real (kind=8) epsCurr
        
        TCurr=TimeCurr
        TEnd=TimeE
        dt3D=TEnd-TCurr !Single time step!
        
        !TIME MARCHING - until end of global time step
        DO WHILE(TCurr.lt.TEnd-eps_small)
                        
            !Calculate solution in time t+dt3D
            TCurr=TCurr+dt3D
            epsCurr=2*epsNonLin
            iter=0  
            !NONLINEAR ITERATIONS   
            !do while(epsCurr.gt.epsNonLin.OR.iter.le.1)
            do while(epsCurr.gt.epsNonLin)
                    !Incremental increase of rain flux
                    if(epsCurr.lt.epsFR3D) then
                        if(TCurr.gt.1200.d0+eps_small.AND.TCurr.lt.1200.d0+eps_small+dt3D) then !Only used for 20181011 experiment (M+C2+R)
                            FluxRelaxation3D=min(1.d0,FluxRelaxation3D+cfFR3D)
                            !if(iter.ne.0) FluxRelaxation3DTemp=min(1.d0,FluxRelaxation3DTemp+cfFR3D)
                            !if(iterKFM.eq.1.AND.iter.eq.0) FluxRelaxation3DTemp=FluxRelaxation3D
                            call CreateCalcInputParamFile
                        endif
                    endif
                !Read input data - for changing parameters during calaculation
                call ReadCalcInputParameters
                iter=iter+1
                !Calculate matrix solution in current time step
                Qrain=0.d0
                call AssembleMatrixEquations
                call SolveSystemOfEquations
                !Go to next time step if PDE is linear equation
                if(problem.eq.'linear') then
                    !Remember current solution
                    CC_LastIter=CC_t1
                    epsCurr=0.d0
                    exit 
                endif
                !Check for convergence
                call CheckConvergence(epsCurr,IcntEps,CoorEps)
                !Remember current solution
                CC_LastIter=CC_t1
                !Screen record
                if(iter.eq.1) write(*,'(a,10xa,e20.10)') '3D MATRIX','Time:',TCurr*TimeUnitConvert
                write(*,'(i6,e20.10,i7,2e20.10,4f12.7,i5)') iter,epsCurr,IcntEps,CoeffEps3D,ValueEps3D,FluxRelaxation3D,xCenter(CoorEps(1)),yCenter(CoorEps(2)),zCenter(CoorEps(3)),SoilType(CoorEps(1),CoorEps(2),CoorEps(3))
                if(iter.eq.1) write(lunFScreen,'(a,10xa,e20.10)') '3D MATRIX','Time:',TCurr*TimeUnitConvert
                write(lunFScreen,'(i6,e20.10,i7,2e20.10,4f12.7,i5)') iter,epsCurr,IcntEps,CoeffEps3D,ValueEps3D,FluxRelaxation3D,xCenter(CoorEps(1)),yCenter(CoorEps(2)),zCenter(CoorEps(3)),SoilType(CoorEps(1),CoorEps(2),CoorEps(3))
                !Exit if maximum number of iterations has been reached
                if(iter.ge.maxiter3D) exit
                !!Reduce under-relaxation when close to true solution
                !if(epsCurr.lt.epsUnderRelax) then
                !    ImplicitUnderRelax=min(1.d0,ImplicitUnderRelax*1.2d0)
                !    ExplicitUnderRelax=min(1.d0,ExplicitUnderRelax*1.2d0)
                !    call CreateCalcInputParamFile
                !endif
                !!Incremental increase of rain flux
                !if(epsCurr.lt.epsFR3D) then
                !    FluxRelaxation3D=min(1.d0,FluxRelaxation3D*cfFR3D)
                !    call CreateCalcInputParamFile
                !endif
                if(iter.eq.resultsIter3D) call ResultsRE3D
            enddo
            
            !Convergence criteria for KFM
            epsCurrKFM=max(epsCurr,epsCurrKFM)
            epsCurr3D=epsCurr
            iterRE3D=iter
            
            !!Set current solution as initial codition for next time step
            !CC_t0=CC_t1
            !Multipe time steps need coordinate CC_t0=CC_t1 with KFM loop 
    
        ENDDO

    end subroutine

    subroutine UpdateCoeffRE3D(sc)
    
        character(*) sc

        SELECT CASE(sc)
            
        CASE('TimeStepEnd')
            CC_t0=CC_t1
        CASE('TimeStepRestart')
            CC_LastIter=CC_t0
        END SELECT

    end subroutine
    
    subroutine ModifyConduitPosition(x1p,x2p,yp,zp)
    !Set conduit in the middle of the closest 3D matrix CVs.

!NOT USED!
    
        integer (kind=4) ixPipe1,ixPipe2,iyPipe,izPipe
        real(kind=8) x1p,x2p,yp,zp
        
        !Karst conduit position
        ixPipe1=nint((x1p-dlx1)/dx)
        ixPipe2=nint((x2p-dlx1)/dx)
        iyPipe=nint((yp-dly1)/dy)
        izPipe=nint((zp-dlz1)/dz)
        
        !Modify karst conduit position based on grid possition
        x1p=xCenter(ixPipe1)
        x2p=xCenter(ixPipe2)
        yp=yCenter(iyPipe)
        zp=zCenter(izPipe)
        !x1p=dlx1+ixPipe1*dx!-dx/2.d0
        !x2p=dlx1+ixPipe2*dx!+dx/2.d0
        !yp=dly1+iyPipe*dy-dy/2.d0   !Check for Fup1
        !zp=dlz1+izPipe*dz-dz/2.d0        
        
    
    end subroutine

    subroutine ResultsRE3D
    
        call WriteResults('Solution')
        if(WritePiezometricResults) call WriteResults('PiezometersHead')
        if(WritePiezometricResults) call WriteResults('SteadyPiezometersHead')
        call CalculateMatrixDischarge
        call ScreenRecord('Discharge')
        !if(abs(TimeCurr-TimeEnd).lt.eps_small) call WriteResults('VauclinExperimentalData')

    end subroutine


!!!  Write concentration results at the current time step

    subroutine ResultsTransport3D
    
        call WriteResults('Transport_Solution')
        if(WritePiezometricResults) call WriteResults('PiezometersConcentration')

    end subroutine
 
     
    real (kind=8) function Hm3D(xp,yp,zp)
    !Matrix head solution.

        real(kind=8) xp,yp,zp
        
        Hm3D=CXYZ(xp,yp,zp,0,0,0,CC_LastIter)

    end function
    
    real (kind=8) function Conc_3D(xp,yp,zp)
    !Matrix concentration solution from the last iteration.

        real(kind=8) xp,yp,zp
        
        Conc_3D=CXYZ(xp,yp,zp,0,0,0,CCC_LastIter)

    end function
    
    real (kind=8) function Conc_3D_Old(xp,yp,zp)
    !Matrix concentration solution from the begining of the time step - t.

        real(kind=8) xp,yp,zp
        
        Conc_3D_Old=CXYZ(xp,yp,zp,0,0,0,CCC_t0)

    end function


    real (kind=8) function Vm3D(xp,yp,zp,mdx,mdy,mdz)
    !Matrix velocity solution.
    
        integer(kind=4) mdx,mdy,mdz
        real(kind=8) xp,yp,zp
        
        Vm3D=MatrixVelocity(xp,yp,zp,mdx,mdy,mdz,CC_LastIter)

    end function
  
!___________________________________________________________!
!INPUT DATA
!!!  Read input data for 3-D matrix
!!!  Read 3-D Gid mesh and other relevant parameters for CV-IGA

    subroutine InputData

        !Number of CVs in each direction
        
        !nx=80
        !ny=51
        !nz=40
        
        !SCF=2
        !nx=20*SCF   !80
        !ny=10*SCF   !51   !10*SCF   !51
        !nz=10*SCF   !40
        
        !SCF=2
!        nx=40   !80
!        ny=25   !51   !10*SCF   !51
!        nz=20   !40

!!!   Read 1-D/3-D Gid mesh parameters
         
         read(lunGid_dat,*) npoin1, nelem1, npoin3, nelem3, nmat

!!!    Read 1-D conduit mesh

          do i = 1, nelem1
          read(lunGid_dat,*) k, ElemsMat1(k), ElemsConec1(k,1), ElemsConec1(k,2)                
          end do

          do i = 1, npoin1
          read(lunGid_dat,*) k, NodesCoord1(k,1), NodesCoord1(k,2), NodesCoord3(k,3)                
          end do
          
!!!    Read 3-D matrix mesh

          do i = 1, nelem3
          read(lunGid_dat,*) k, ElemsMat3(k), ElemsConec3(k,1), ElemsConec3(k,2), ElemsConec3(k,3), ElemsConec3(k,4)                
          end do

          do i = 1, npoin3
          read(lunGid_dat,*) k, NodesCoord3(k,1), NodesCoord3(k,2), NodesCoord3(k,3)                
          end do
          
          
 !!!   Number of 3-D matrix control volumes in each direction
                 
         read(lunGid_dat,*)nx,ny,nz
         
        !!SCF=2
        !nx=15   !80
        !ny=15   !51   !10*SCF   !51
        !nz=15   !40

        
!        !!Vauclin et al. 1979.
!        !SatCond=9.72d-5
!        !AnisotropZ=1.d0,1.d0,1.d0
!        !Porosity=0.3d0
!        !ResidualSaturation=0.01d0
!        !VGM_alpha=3.3d0
!        !VGM_n=4.1d0
!        !VGM_m=1.d0-1.d0/VGM_n(1)
!        !VGM_tau=0.5d0
!        !SpecificStorage=0.d0
!        !ResidPress=-20.5d0


        !Material soil properties - CALIBRATION - M - Hyprop-Fit
        UnsatSoilModel(1)='VGM_C_B'  !traditional-constrained: 'VGM_C'   ;   bimodal-constrained: 'VGM_C_B'
        UnsatSoilModel(2)='VGM_C_B'
        UnsatSoilModel(3)='VGM_C'        
        !
        !SatCond=(/6.d-3, 2.0d-4, 6.d-2/)
        !AnisotropZ=(/1.d0,1.d0,1.d0/) !Anisotropy coefficient in z direction
        !SpecificStorage=(/1.d-5, 1.d-4, 1.d-6/) 
        !!
        !VGM_alpha(:,1)=(/0.2189d2,  0.0288d2, 0.1d2/)  !0.594d2
        !VGM_alpha(:,2)=(/0.00238d2, 0.0181d2, 0.d0/)
        !VGM_n(:,1)=(/3.901d0, 7.862d0, 2.4d0/)   !1.2199d0
        !VGM_n(:,2)=(/1.321d0, 1.572d0, 0.d0/)
        !VGM_m=1.d0-1.d0/VGM_n   !Constrained VGM model
        !VGM_w(:,2)=(/0.38d0, 0.33d0, 0.0d0/)
        !VGM_w(:,1)=1.d0-VGM_w(:,2)
        !VGM_tau=(/2.960d0, 2.295d0, 0.5d0/)
        !Porosity=(/0.325d0, 0.38d0, 0.42d0/)
        !ResidualWaterContent=(/0.005d0, 0.02d0, 0.005d0/)
        !ResidualSaturation=ResidualWaterContent/Porosity    !(/0.36d0, 0.081d0, 0.0d0/)
        !ResidPress=-2.5d0

        do i=1,nmat
        read(lunGid_dat,*)SatCond(i),AnisotropZ(i),SpecificStorage(i),VGM_alpha(i,1),  &
          VGM_alpha(i,2),VGM_n(i,1),VGM_n(i,2),VGM_w(i,2),VGM_tau(i),Porosity(i),   &
          ResidualWaterContent(i)
          
          
!        SatCond=(/3.4d-3, 2.0d-4, 6.0d-2/)  !(/6.0d-3, 1.0d-4, 6.0d-2/)  !(/2.8d-3, 3.7d-4, 6.0d-2/)  !(/2.8d-3, 3.7d-4, 4.0d-2/)     (/3.2d-3, 3.0d-4, 4.0d-2/)  !(/6.2d-3, 2.0d-4, 4.0d-2/)   !(/3.2d-3, 3.0d-4, 4.0d-2/)  !(/3.1d-3, 3.1d-4, 2.5d-2/)
!       AnisotropZ=(/2.7d0,2.5d0,1.d0/) !Anisotropy coefficient in z direction
!        SpecificStorage=(/1.d-5, 1.d-4, 1.d-6/) 
!        VGM_alpha(:,1)=(/0.18d2,  0.03d2, 0.1d2/)  !(/0.22d2,  0.03d2, 0.1d2/)
!        VGM_alpha(:,2)=(/0.005d2, 0.02d2, 0.0d2/)
!        VGM_n(:,1)=(/3.2d0, 9.2d0, 2.9d0/)   !(/3.2d0, 9.2d0, 2.9d0/)
!        VGM_n(:,2)=(/1.3d0, 2.2d0, 0.0d0/)
        VGM_m=1.d0-1.d0/VGM_n   !Constrained VGM model
!        VGM_w(:,2)=(/0.38d0, 0.33d0, 0.0d0/)
        VGM_w(:,1)=1.d0-VGM_w(:,2)
!        VGM_tau=(/2.960d0, 2.295d0, 0.5d0/)
!        Porosity=(/0.325d0, 0.38d0, 0.40d0/)
!        ResidualWaterContent=(/0.005d0, 0.02d0, 0.01d0/)    !(/0.005d0, 0.02d0, 0.001d0/)

         end do

        ResidualSaturation=ResidualWaterContent/Porosity
        ResidPress=-2.5d0
        
        !Boundary flow conditions: Neumann/Dirichlet/Reservoir/Outflow
        BCType(1)='Reservoir'   !East - downstream
        BCType(2)='Reservoir'   !West - upstream
        BCType(3)='Neumann'     !North
        BCType(4)='Neumann'     !South
        BCType(5)='Neumann'     !Top
        BCType(6)='Neumann'     !Bottom
        !Model only half of domain
        if(Symmetry3D.eq.'yes') BCType(2)='Neumann'
        read(lunGid_dat,*)BCType(1), BCType(2), BCType(3), BCType(4), BCType(5), BCType(6) 
        read(lunGid_dat,*)BCValue(1), BCValue(2), BCValue(3), BCValue(4), BCValue(5), BCValue(6) 
        
        !CALCULATION FLOW PARAMETERS
        
        !Time integration parameter
        thetaRE=1.d0    !Currently not changeable
        !Linear system of equations solver
        LinSol='paradiso'       !simq   paradiso
        !Write system of equations Ax=b into data file
        writeSoE='no'
        !Linear or nonlinear problem
        problem='nonlinear'     !linear/nonlinear
        !Maximum number of nonlinear iterations
        !maxiter3D -> GlobalData
        !Nonlinear solver
        epsNonLin=epsMatrix3D
        CheckResidual='no'
        !UnderRelax=1.d0 - No Under-Relaxation - Usefull to correct UR factor after 
        ImplicitUnderRelax=IUR3D
        ExplicitUnderRelax=EUR3D
        !Reduce under-relaxation when close to true solution
        epsUnderRelax=epsUR3D
        !Use average CV results for matrix assembly or perform full integration
        FullIntegration='no'
        !Trapezoid rule integration parameter
        kmax=6
        kmaxCI1D=7
        kmax3D=1
        kmaxMV=3

!!!   Material transport parameters
        
        read(lunGid_dat,*)Diffusion_Matrix, Alfa_l,Alfa_t
        
!!!   Transport boundary conditions        
        
        read(lunGid_dat,*)BCType_Transport(1), BCType_Transport(2), BCType_Transport(3), &
                          BCType_Transport(4), BCType_Transport(5), BCType_Transport(6) 
        read(lunGid_dat,*)BCValue_Transport(1),BCValue_Transport(2),BCValue_Transport(3), &
                          BCValue_Transport(4),BCValue_Transport(5),BCValue_Transport(6) 
        


        
            
    end subroutine

    subroutine CreateCalcInputParamFile

        open(lunF1,file='Input3D.inp')
        write(lunF1,*) 'maxiter3D',maxiter3D
        write(lunF1,*) 'epsNonLin',epsNonLin
        write(lunF1,*) 'ImplicitUnderRelax',ImplicitUnderRelax
        write(lunF1,*) 'ExplicitUnderRelax',ExplicitUnderRelax
        write(lunF1,*) 'epsUnderRelax',epsUnderRelax
        write(lunF1,*) 'FluxRelaxation3D',FluxRelaxation3D
        !write(lunF1,*) 'FluxRelaxation3DTemp',FluxRelaxation3DTemp
        write(lunF1,*) 'epsFR3D',epsFR3D
        write(lunF1,*) 'cfFR3D',cfFR3D
        write(lunF1,*) 'resultsIter3D',resultsIter3D
        close(lunF1)
        
    end subroutine

    subroutine ReadCalcInputParameters
    
        character(len=40) ch,CSwitch
    
        !Control switch
        open(lunF1,file='CSwitch3D.inp')
        read(lunF1,*) CSwitch
        close(lunF1)
        if(CSwitch.eq.'no') return
        
        write(*,*) 'Change parameters and press enter to continue.'
        write(lunFScreen,*) 'Change parameters and press enter to continue.'
        read(*,*)
        
        !Read parameters values
        open(lunF1,file='Input3D.inp')
        read(lunF1,*) ch,maxiter3D
        read(lunF1,*) ch,epsNonLin
        read(lunF1,*) ch,ImplicitUnderRelax
        read(lunF1,*) ch,ExplicitUnderRelax
        read(lunF1,*) ch,epsUnderRelax
        read(lunF1,*) ch,FluxRelaxation3D
        !read(lunF1,*) ch,FluxRelaxation3DTemp
        read(lunF1,*) ch,epsFR3D
        read(lunF1,*) ch,cfFR3D
        read(lunF1,*) ch,resultsIter3D
        close(lunF1)
        
        write(*,*) 'maxiter3D',maxiter3D
        write(*,*) 'epsNonLin',epsNonLin
        write(*,*) 'ImplicitUnderRelax',ImplicitUnderRelax
        write(*,*) 'ExplicitUnderRelax',ExplicitUnderRelax
        write(*,*) 'epsUnderRelax',epsUnderRelax
        write(*,*) 'FluxRelaxation3D',FluxRelaxation3D
        !write(*,*) 'FluxRelaxation3DTemp',FluxRelaxation3DTemp
        write(*,*) 'epsFR3D',epsFR3D
        write(*,*) 'cfFR3D',cfFR3D
        write(*,*) 'resultsIter3D',resultsIter3D
        
        write(lunFScreen,*) 'maxiter3D',maxiter3D
        write(lunFScreen,*) 'epsNonLin',epsNonLin
        write(lunFScreen,*) 'ImplicitUnderRelax',ImplicitUnderRelax
        write(lunFScreen,*) 'ExplicitUnderRelax',ExplicitUnderRelax
        write(lunFScreen,*) 'epsUnderRelax',epsUnderRelax
        write(lunFScreen,*) 'FluxRelaxation3D',FluxRelaxation3D
        !write(lunFScreen,*) 'FluxRelaxation3DTemp',FluxRelaxation3DTemp
        write(lunFScreen,*) 'epsFR3D',epsFR3D
        write(lunFScreen,*) 'cfFR3D',cfFR3D
        write(lunFScreen,*) 'resultsIter3D',resultsIter3D
                
        
        !Return CSwitch to 'no'
        open(lunF1,file='CSwitch3D.inp')
        write(lunF1,*) 'no'
        close(lunF1)        
        
    
    end subroutine

!Function which calculates backward position of particle from advective step

    real (kind=8) function PTrack_Matrix(xp,yp,zp)
    
    
        real (kind=8) xp,yp,zp,vpx,vpy,vpz,Por
!        integer (kind=8) ip,jp,kp,isoil
        
        
!          ip=idnint((xp-0.5d0*dx-dlx1)/(dx))
!          jp=idnint((yp-0.5d0*dy-dly1)/(dy))                
!          kp=idnint((zp-0.5d0*dz-dlz1)/(dz))          
          
!        isoil = SoilType(ip,jp,kp)
!        Por = Porosity(isoil)
        vpx = Vm3D(xp,yp,zp,1,0,0)   !/Por
        vpy = Vm3D(xp,yp,zp,0,1,0)   !/Por
        vpz = Vm3D(xp,yp,zp,0,0,1)   !/Por
        
        xp = xp - vpx*dt3D
        if (xp.lt.dlx1) xp=dlx1
        if (xp.gt.dlx2) xp=dlx2
        yp = yp - vpy*dt3D
        if (yp.lt.dly1) yp=dly1
        if (yp.gt.dly2) yp=dly2       
        zp = zp - vpz*dt3D
        if (zp.lt.dlz1) zp=dlz1
        if (zp.gt.dlz2) zp=dlz2
        
        Ptrack_Matrix = Conc_3D(xp,yp,zp)

    end function
      


!Initial and boundary conditions:

    real (kind=8) function IC_Matrix(xp,yp,zp)
    !Initial condition for hydraulic head in porous matrix.
    
        real (kind=8) xp,yp,zp
        real (kind=8) hU,hD

        !hU=ReservoirLevels(2)
        !hD=ReservoirLevels(1)
        !IC_Matrix=hD+(hU-hD)/(dlx1-dlx2)*(xp-dlx2)
        IC_Matrix=ReservoirLevels(2)

    end function
      
    real (kind=8) function ReservoirLevels(side)
    !Returns values of piezometric head in upstream and downstream reservoirs &
    !for specified time (time dependent boundary conditions).
    
        integer (kind=4) side
        real (kind=8) zp,tp,hU,hD,delT
        !character (len=*) res
        
        
        if(TestCase.eq.'C3') then
            hU=1.555d0              !1.661d0    !1.558d0   !1.472d0   !1.660d0   !1.45d0   !1.62d0!+2.d0
            hD=1.463d0              !1.477d0    !1.465d0   !1.472d0   !1.477d0   !1.32d0   !1.54d0!+2.d0
            !delT=(dt3D*20.d0)
        elseif(TestCase.eq.'C2') then
            hU=1.515d0    !1.5045d0 
            hD=1.351d0    !1.3515d0             
            !delT=(dt3D*20.d0) 
        elseif(TestCase(1:2).eq.'MA') then
            if(TestCase.eq.'MA01') then
                hU=1.454d0 
                hD=1.453d0
            elseif(TestCase.eq.'MA02') then
                hU=1.455d0 
                hD=1.450d0                
            elseif(TestCase.eq.'MA03') then
                hU=1.453d0 
                hD=1.452d0                
            elseif(TestCase.eq.'MA04') then
                hU=1.453d0 
                hD=1.450d0                
            elseif(TestCase.eq.'MA05') then
                hU=1.453d0 
                hD=1.452d0                
            elseif(TestCase.eq.'MA06') then
                hU=1.454d0 
                hD=1.289d0                
            elseif(TestCase.eq.'MA07') then
                hU=1.453d0 
                hD=1.288d0
            elseif(TestCase.eq.'MA08') then
                hU=1.455d0 
                hD=1.290d0  
            elseif(TestCase.eq.'MA09') then
                hU=1.455d0 
                hD=1.290d0  
            elseif(TestCase.eq.'MA10') then
                hU=1.455d0 
                hD=1.290d0  
            elseif(TestCase.eq.'MA11') then
                hU=1.455d0 
                hD=1.290d0   
            endif
        elseif(TestCase.eq.'RI01') then
            hU=1.471d0 
            hD=1.469d0             
        endif
        
        !Upstream
        if(side.eq.2) then
            ReservoirLevels=hU
        !Downstream
        elseif(side.eq.1) then
            !Linear decreasing of downstream water level
            ReservoirLevels=hD  !-(hU-hD)/delT*(tp-TStart+dt3D)+hU
        endif


    end function
      
    real (kind=8) function FluxBC(side,xp,yp,zp)
    !Returns values of flux [m/s] on sandbox boundaries with Neumann boundary conditions.
    
    
        integer (kind=4) side
        real (kind=8) xp,yp,zp,qFlux,qFluxShower
        real (kind=8) Q1,Q2,T1,T2
        real (kind=8) tp

        FluxBC=0.d0
        qFlux=0.d0
        qFluxShower=0.d0
        
        
        !Test uniform rain - qFlux
        !Test surface injection - qFluxShower
        if(TestCase.eq.'Test16') then
            if(TCurr.gt.3200.d0.and.TCurr.lt.4400.d0) then 
            qFlux=-20.0d0d0      !2 l/min/m2 ;  Area = 10 m2; Rain duration = 20 min.
            qFlux=qFlux/DischargeUnitConvert/((dlx2-dlx1)*(dly2-dly1))
            end if
        end if
!        if(TestCase.eq.'C3') then
!            qFlux=-10.17d0      !-11.32d0
!            qFlux=qFlux/DischargeUnitConvert/((dlx2-dlx1)*(dly2-dly1))
!        elseif(TestCase.eq.'C2') then
!            !Sprinklers
!            qFlux=-11.0d0    !-11.8d0
!            !if(TCurr.gt.3600.d0+eps_small) qFlux=-11.d0
!            qFlux=qFlux/DischargeUnitConvert/((dlx2-dlx1)*(dly2-dly1))
!            !Shower heads
!            qFluxShower=-42.3d0/1.026d0     !*1.00653413d0     !/1.115625d0    !-44.8d0/1.115625d0!*0.5d0    !Correction because discretization
!            qFluxShower=qFluxShower/DischargeUnitConvert/(0.9d0*0.9d0)  !!!***    
!        elseif(TestCase.eq.'RI01') then
!            qFlux=-11.7d0
!            qFlux=qFlux/DischargeUnitConvert/((dlx2-dlx1)*(dly2-dly1))
!        endif
        
        if(side.eq.5) then
!            if(TestCase.eq.'C3') then
!                if(TCurr.lt.1200.d0+eps_small.OR.TCurr.gt.4800.d0+eps_small) return
!                FluxBC=-qFlux*FluxRelaxation3D  !FluxRelaxation3DTemp
!            elseif(TestCase.eq.'C2') then
!                if(TCurr.lt.1200.d0+eps_small.OR.TCurr.gt.5400.d0+eps_small) return
!                FluxBC=-qFlux*FluxRelaxation3D
!                if(TCurr.lt.3600.d0+eps_small.OR.TCurr.gt.5400.d0+eps_small) return
!                if(xp.lt.1.3d0.OR.xp.gt.2.2d0) return   !!!***  (xp.lt.1.1d0.OR.xp.gt.1.90)  (xp.lt.1.2d0.OR.xp.gt.1.8d0)
!                if(yp.lt.0.8d0.OR.yp.gt.1.7d0) return   !!!***  (yp.lt.0.8d0.OR.yp.gt.1.6d0)  (yp.lt.0.9d0.OR.yp.gt.1.5d0) 
!                FluxBC=-(qFlux+qFluxShower)*FluxRelaxation3D  !FluxRelaxation3DTemp
!            elseif(TestCase.eq.'RI01') then
!                if(TCurr.gt.1800.d0+eps_small) return
!                FluxBC=-qFlux*FluxRelaxation3D
!            endif                
         FluxBC=-(qFlux+qFluxShower)*FluxRelaxation3D  !FluxRelaxation3DTemp

        endif
        
    end function
    
    real (kind=8) function SourceTerm(xp,yp,zp)
    !Source term for matrix equation
    
        real (kind=8) xp,yp,zp
        real (kind=8) xPac,yPac,zPac,Qs
        integer (kind=4) ip,ip1,jp,jp1,kp,kp1

        SourceTerm = 0.d0
        Qs = 0.0d0
        !if(TCurr.gt.(TimeEnd+eps_small).OR.Tcurr.lt.2.5*TimeStart) return
        !if(xp.eq.xVertex(nx/2).AND.yp.eq.yVertex(ny/2).AND.zp.eq.zVertex(nz/2)) then
        !    SourceTerm=0.d-3
        !endif
        
        !xPac=2.22d0; yPac=1.13d0; zPac=0.97d0  !P12
        !xPac=1.99d0; yPac=1.74d0; zPac=0.74d0  !P16
        !xPac=1.52d0; yPac=0.62d0; zPac=0.84d0   !P17
        !Qs=40.d0
        
!        if(TestCase(1:2).eq.'MA') then
!            if(TestCase.eq.'MA01') then
!                xPac=2.36d0; yPac=1.10d0; zPac=0.74d0  !P12
!                Qs=32.3d0
!            elseif(TestCase.eq.'MA02') then
!                xPac=2.04d0; yPac=1.72d0; zPac=0.76d0  !P16
!                Qs=32.4d0
!            elseif(TestCase.eq.'MA03') then
!                xPac=2.31d0; yPac=1.74d0; zPac=0.87d0  !P13
!                Qs=32.4d0
!            elseif(TestCase.eq.'MA04') then
!                xPac=1.50d0; yPac=0.58d0; zPac=0.87d0  !P17
!                Qs=33.3d0
!            elseif(TestCase.eq.'MA05') then
!                xPac=2.41d0; yPac=0.76d0; zPac=0.62d0  !P11
!                Qs=32.3d0
!            endif
            
 !           if(xp+0.5d0*dx.gt.xPac.AND.xp-0.5d0*dx.lt.xPac) then
 !               if(yp+0.5d0*dy.gt.yPac.AND.yp-0.5d0*dy.lt.yPac) then
 !                   if(zp+0.5d0*dz.gt.zPac.AND.zp-0.5d0*dz.lt.zPac) then
 !                       SourceTerm=Qs/DischargeUnitConvert/(dx*dy*dz)
 !                   endif
 !               endif
 !           endif
 !       endif
 
          ip1=idnint((x_injection-0.5d0*dx-dlx1)/(dx))
          jp1=idnint((y_injection-0.5d0*dy-dly1)/(dy))                
          kp1=idnint((z_injection-0.5d0*dz-dlz1)/(dz))          
          
        
          if(TestCase.eq.'Test16') then
          ip=idnint((xp-0.5d0*dx-dlx1)/(dx))
          jp=idnint((yp-0.5d0*dy-dly1)/(dy))                
          kp=idnint((zp-0.5d0*dz-dlz1)/(dz))          
!!!  200 litre of salt water injected in 11 minutes          
          if (ip.eq.ip1.and.jp.eq.jp1.and.kp.eq.kp1) then
          if(TCurr.gt.2000.0d0.and.Tcurr.le.2660.0d0) Qs = 200.0d0/11.0d0   
          end if
          end if


           SourceTerm=Qs/DischargeUnitConvert/(dx*dy*dz)

    end function

!Soil Characteristic Functions:

    real (kind=8) function EffectiveSaturation(it,pp,S)
    !This function returns matrix effective saturation value as function of pressure head.
    !pp-pressure
    !S(2)-two modes for bimodal VGM model.
    !Van Genuchten-Mualem model.
    
        integer (kind=4) it,ii
        real (kind=8) pp,S(2)
        
        !Return when p>0
        if(pp.ge.0.d0) then
            EffectiveSaturation=1.d0
            S=1.d0
            return
        endif
        
        select case(UnsatSoilModel(it))
        
        !Traditional constrained Van Genuchten-Mualem model.
        case('VGM_C')
            EffectiveSaturation=1.d0/((1.d0+(abs(VGM_alpha(it,1)*pp))**VGM_n(it,1))**VGM_m(it,1))
        !Bimodal constrained Van Genuchten-Mualem model.    
        case('VGM_C_B')
            !Calculate two modes of Se
            do ii=1,2
                S(ii)=(1.d0/((1.d0+(abs(VGM_alpha(it,ii)*pp))**VGM_n(it,ii))**VGM_m(it,ii)))
            enddo
            EffectiveSaturation=VGM_w(it,1)*S(1)+VGM_w(it,2)*S(2)
        end select
    
    end function 

    real (kind=8) function Saturation(it,Se)
    !This function returns matrix saturation value.
    !Se-effective saturation.
    
        integer (kind=4) it
        real (kind=8) Se

        Saturation=ResidualSaturation(it)+Se*(1.d0-ResidualSaturation(it))
        
    end function 

    real (kind=8) function SpecMoistCapacity(it,Se,S)
    !This function returns d/dp(WaterContent) value.
    !Se-effective saturation.
    !Van Genuchten-Mualem model.
        
        integer (kind=4) it,ii
        real (kind=8) Se,S(2),C(2)
        
        select case(UnsatSoilModel(it))
        
        !Traditional constrained Van Genuchten-Mualem model.
        case('VGM_C')
            SpecMoistCapacity=(Porosity(it)*VGM_m(it,1)*VGM_alpha(it,1)*(1.d0-ResidualSaturation(it)))/(1.d0-VGM_m(it,1))*Se**(1.d0/VGM_m(it,1))*(1.d0-(Se)**(1.d0/VGM_m(it,1)))**VGM_m(it,1)
        !Bimodal constrained Van Genuchten-Mualem model.    
        case('VGM_C_B')
            !Calculate two modes of C
            do ii=1,2
                C(ii)=(Porosity(it)*VGM_m(it,ii)*VGM_alpha(it,ii)*(1.d0-ResidualSaturation(it)))/(1.d0-VGM_m(it,ii))*S(ii)**(1.d0/VGM_m(it,ii))*(1.d0-(S(ii))**(1.d0/VGM_m(it,ii)))**VGM_m(it,ii)
            enddo
            SpecMoistCapacity=VGM_w(it,1)*C(1)+VGM_w(it,2)*C(2)
        end select
        
    end function 
           
    real (kind=8) function RelativePermeability(it,Se,S)
    !This function returns relative permeability value.
    !Se-effective saturation
    !S(2)-two modes for bimodal VGM model.
    !Van Genuchten-Mualem model.
    
        integer (kind=4) it,ii
        real (kind=8) Se,S(2)
        real (kind=8) KP(3)
        
        
        select case(UnsatSoilModel(it))
            
        case('VGM_C')
            RelativePermeability=Se**(VGM_tau(it))*(1.d0-(1.d0-Se**(1.d0/VGM_m(it,1)))**VGM_m(it,1))**2
        case('VGM_C_B')
            KP=0.d0
            do ii=1,2
                KP(1)=KP(1)+VGM_w(it,ii)*S(ii)
                KP(2)=KP(2)+VGM_w(it,ii)*VGM_alpha(it,ii)*(1.d0-(1.d0-S(ii)**(1.d0/VGM_m(it,ii)))**VGM_m(it,ii))
                KP(3)=KP(3)+VGM_w(it,ii)*VGM_alpha(it,ii)
            enddo
            RelativePermeability=(KP(1))**VGM_tau(it)*(KP(2)/KP(3))**2
        end select
                
    end function 
        
    real (kind=8) function WaterContent(it,Se)
    !This function returns matrix water conten value.
    !Se-effective saturation.
    
        integer (kind=4) it
        real (kind=8) Se

        WaterContent=Porosity(it)*Saturation(it,Se)
        
        
    end function                
    
!___________________________________________________________!
!PRIVATE SUBROUTINES

    subroutine CalculateParameters
    
        integer (kind=4) icnt,ii,jj,kk
        !integer (kind=4) indx,indy,indz,indl,nzx,nzy,nzz,nzl,CNRDer
!real*8 xx

        !Number of segments of basis functions
        if(BasisFun.eq.'fup')    nSegments=nOrder+2
        if(BasisFun.eq.'spline') nSegments=nOrder+1
        !Number of external basis functions
        if(BasisFun.eq.'fup')    nExternal=(nOrder+1)/2
        if(BasisFun.eq.'spline') nExternal=(nOrder)/2
        
        !Domain boundaries
        dlx1=x1Matrix   ;   dlx2=x2Matrix
        dly1=y1Matrix   ;   dly2=y2Matrix
        dlz1=z1Matrix   ;   dlz2=z2Matrix

        !Calculate dimensions of CV
        dx=(dlx2-dlx1)/dfloat(nx)
        dy=(dly2-dly1)/dfloat(ny)
        dz=(dlz2-dlz1)/dfloat(nz)

        !Total number of 3D basis functions = total number of CVs
        nxTOT=(nx+2*nExternal)
        nyTOT=(ny+2*nExternal)
        nzTOT=(nz+2*nExternal)
        NTOT3D=nxTOT*nyTOT*nzTOT
 
        !Allocate memory
        allocate(xVertex(-nExternal:nx-1+nExternal),yVertex(-nExternal:ny-1+nExternal),zVertex(-nExternal:nz-1+nExternal))
        allocate(xCenter(-nExternal:nx-1+nExternal),yCenter(-nExternal:ny-1+nExternal),zCenter(-nExternal:nz-1+nExternal))
        allocate(xcp1(-nExternal:nx-1+nExternal),ycp1(-nExternal:ny-1+nExternal),zcp1(-nExternal:nz-1+nExternal))
        allocate(xcp2(-nExternal:nx-1+nExternal),ycp2(-nExternal:ny-1+nExternal),zcp2(-nExternal:nz-1+nExternal))
        allocate(indx(-nExternal:nx-1+nExternal),indy(-nExternal:ny-1+nExternal),indz(-nExternal:nz-1+nExternal))
        allocate(SoilType(-nExternal:nx-1+nExternal,-nExternal:ny-1+nExternal,-nExternal:nz-1+nExternal))
        allocate(CC_t0(NTOT3D),CC_LastIter(NTOT3D),CC_t1(NTOT3D))
        allocate(CNL(-2:2),CNR(-2:2))
        allocate(DELX(-2:2),DELY(-2:2),DELZ(-2:2))        

        !Define finite volume geometry for each direction
        do ii=-nExternal,nx-1+nExternal
            call FVGeometry(ii,nx,dlx1,dlx2,dx,xVertex(ii),xCenter(ii),xcp1(ii),xcp2(ii),indx(ii))
        enddo
        do jj=-nExternal,ny-1+nExternal
            call FVGeometry(jj,ny,dly1,dly2,dy,yVertex(jj),yCenter(jj),ycp1(jj),ycp2(jj),indy(jj))
        enddo
        do kk=-nExternal,nz-1+nExternal
            call FVGeometry(kk,nz,dlz1,dlz2,dz,zVertex(kk),zCenter(kk),zcp1(kk),zcp2(kk),indz(kk))
        enddo
        
    !!Trial solution space
    !open(100,file='BFUNspace.dat')
    !!Coordinates of 3D basis functions vertices
    !do icnt=-nExternal,nx-1+nExternal
    !    xVertex(icnt)=dlx1+dfloat(icnt)*dx+dx/2.d0      
    !    write(100,'(a,a,i2)') 'ZONE T=','K',icnt+11
    !    do xx=dlx1,dlx2+eps_small,dx*0.01d0 
    !        !write(*,*) icnt,xx
    !        write(100,'(2e20.10)') xx,BFUN(norder, xVertex(icnt), xx, dx, 0, dlx1, dlx2)
    !    enddo
    !enddo
    !close(100)  
    !stop
        
        !Calculate total number of non-zero matrix coefficients.
        NNZ=NTOT3D*3**3
        
        !Allocate memory
        if(LinSol.eq.'simq') then
            allocate(ASIMQ(NTOT3D,NTOT3D))
            ASIMQ=0.d0
        elseif(LinSol.eq.'paradiso') then
            allocate(APARADISO(NNZ),IA(NTOT3D+sup1),JA(NNZ))
            APARADISO=0.d0
        endif
        
        !Number of non-zeros basis function left and right in each direction depending of ind(x/y/z)
        !Prepared for Fup1 or spline B2; CL(ind)=value
        CNL(-2:2)=(/1,1,1,1,1/)
        CNR(-2:2)=(/1,1,1,1,1/)
        
        !DELXYZ must multiply BFIV to obtain final integral value; DELX(indx)
        DELX(-2:2)=(/dx,dx,dx,dx,dx/)
        DELY(-2:2)=(/dy,dy,dy,dy,dy/)
        DELZ(-2:2)=(/dz,dz,dz,dz,dz/)
        
        !Write some information on screen
        call ScreenRecord('CalculateParameters')
            
    end subroutine
    
    subroutine CalculateIntegrals
    !This subroutine prepares 1D integrals of basis function over computational CVs.
    !3D integrals is splited as: [int(BFUN(x,y,z))dV]=[int(BFUN(x))dx]*[int(BFUN(y))dy]*[int(BFUN(z))dz] &
    !and used later for assembly of discrete algebraic system of equations.
    !Calculated values BFIV must be multiplied by d(x/y/z) to obtain real integral value.
    !BFIV(min(ind):max(ind),-nSegments:nSegments)
    !Written for Fup1 or B2-spline
    
        integer (kind=4) icnt,ii,mdx,indx,deriv
        real (kind=8) xv,UnitDX
    
        !Allocate array 
        allocate(BFIV(-nExternal-1:nExternal+1,-nSegments/2:nSegments/2))
        
        !Set to zero
        BFIV=0.d0
        !Calculate for unit UnitDX
        UnitDX=1.d0
        !Calculate only integral of function 0th derivative
        mdx=0
        
        !INTERNAL CV: indx=0
        indx=0
        !All neighbor basis function integrals
        do ii=-nSegments/2,nSegments/2
            xv=dfloat(ii)*UnitDX
            call trap_int_1D(bfun,norder,xv,UnitDX,mdx,-1.d9,+1.d9,-0.5d0*UnitDX,0.5d0*UnitDX,kmaxCI1D,BFIV(indx,ii))
        enddo
        
        !MODIFIED INTERNAL CV (ii=0, or ii=nx-1): indx=-1 and indx=1
        indx=1
        !All neighbor basis function integrals
        do ii=-nSegments/2,nSegments/2
            xv=dfloat(ii)*UnitDX
            call trap_int_1D(bfun,norder,xv,UnitDX,mdx,-0.5d0*UnitDX,2.5d0*UnitDX,0.0d0,0.5d0*UnitDX,kmaxCI1D,BFIV(-indx,ii))
            call trap_int_1D(bfun,norder,xv,UnitDX,mdx,-2.5D0*UnitDX,0.5d0*UnitDX,-0.5d0*UnitDX,0.0d0,kmaxCI1D,BFIV(indx,ii))
        enddo
        
        !BOUNDARY CV (ii=-1, or ii=nx): indx=-2 and indx=2
        indx=2
        !All neighbor basis function values
        do ii=-nSegments/2,nSegments/2
            xv=dfloat(ii)*UnitDX
            call trap_int_1D(bfun,norder,xv,UnitDX,mdx,-0.5d0*UnitDX,2.5d0*UnitDX,-0.5d0*UnitDX,0.0d0,kmaxCI1D,BFIV(-indx,ii))
            call trap_int_1D(bfun,norder,xv,UnitDX,mdx,-2.5d0*UnitDX,0.5d0*UnitDX,0.0d0*UnitDX,0.5d0,kmaxCI1D,BFIV(indx,ii))
        enddo
        
        !Partition of unity constant value is not unity for Fup basis functions
        do ii=-nSegments/2,nSegments/2
            xv=dfloat(ii)*UnitDX
            PartOfConstantValue=PartOfConstantValue+bfun(norder,xv,0.d0,UnitDX,mdx,-1.d9,+1.d9)
        enddo
    
    end subroutine

    subroutine DefineSoilType
    !In this subroutine soil type is define for every CV.
    
        integer (kind=4) ii,jj,kk,k,k1,k2,k3,k4,i
        real (kind=8) VSUpstream,VSDownstream   
    
        !Set all CVs to principal material
        SoilType=1 

!!!   Define soil type for all control volumes using the 3-D GiD mesh

!          do i = 1, nelem3
!          k = i
!          k1 = ElemsConec3(k,1)
!          k2 = ElemsConec3(k,2)
!          k3 = ElemsConec3(k,3)
!          k4 = ElemsConec3(k,4)
!          x_ck = (NodesCoord3(k1,1)+NodesCoord3(k2,1)+NodesCoord3(k3,1)+NodesCoord3(k4,1))/4.0d0 
!          y_ck = (NodesCoord3(k1,2)+NodesCoord3(k2,2)+NodesCoord3(k3,2)+NodesCoord3(k4,2))/4.0d0 
!          z_ck = (NodesCoord3(k1,3)+NodesCoord3(k2,3)+NodesCoord3(k3,3)+NodesCoord3(k4,3))/4.0d0 
!          ii=idnint((x_ck-0.5d0*dx-dlx1)/(dx))
!          jj=idnint((y_ck-0.5d0*dy-dly1)/(dy))                
!          kk=idnint((z_ck-0.5d0*dz-dlz1)/(dz))
!          SoilType(ii,jj,kk) = ElemsMat3(k) 
!          
!          end do

        
        !Move vertical screen (1.95-2.21) in x-direction
        VSUpstream=2.125d0
        VSDownstream=1.875d0
        
!RETURN
!****************************************************************!
!
!        !ONLY VERTICAL SCREEN
!        do ii=-nExternal,nx-1+nExternal
!            do jj=-nExternal,ny-1+nExternal
!                do kk=-nExternal,nz-1+nExternal
!                    !Verical screen
!                    if(xCenter(ii).ge.1.95d0.AND.xCenter(ii).lt.2.21d0) SoilType(ii,jj,kk)=2
!                enddo
!            enddo
!        enddo
!
!        !ONLY EPIKARST SCREEN
!        do ii=-nExternal,nx-1+nExternal
!            do jj=-nExternal,ny-1+nExternal
!                do kk=-nExternal,nz-1+nExternal
!                    !Epikarst
!                    if(zCenter(kk).ge.1.75d0) SoilType(ii,jj,kk)=3
!                enddo
!            enddo
!        enddo
!    
!RETURN
!****************************************************************!


        !Define zones of different conducivity
        do ii=-nExternal,nx-1+nExternal
            do jj=-nExternal,ny-1+nExternal
                do kk=-nExternal,nz-1+nExternal
                    !Layer 2
                    if(zCenter(kk).ge.0.25d0.AND.zCenter(kk).lt.0.5d0) then    
                    !if(zCenter(kk).ge.0.25d0.AND.zCenter(kk).lt.0.45d0) then   !FQ-Around conduit
                        !Layer 2a
                        if(xCenter(ii).gt.0.73d0.AND.xCenter(ii).lt.1.33d0) then
                            if(yCenter(jj).gt.0.43d0.AND.yCenter(jj).lt.1.07d0) SoilType(ii,jj,kk)=3
                        endif
                        !Layer 2b
                        if(xCenter(ii).gt.2.83d0.AND.xCenter(ii).lt.3.46d0) then
                            if(yCenter(jj).gt.0.28d0.AND.yCenter(jj).lt.0.87d0) SoilType(ii,jj,kk)=3
                        endif                        
                    endif
                    !Layer 3
                    if(zCenter(kk).ge.0.5d0.AND.zCenter(kk).lt.0.75d0) then
                    !if(zCenter(kk).ge.0.45d0.AND.zCenter(kk).lt.0.75d0) then   !FQ-Around conduit
                        !Layer 3a
                        if(xCenter(ii).gt.VSUpstream.AND.xCenter(ii).lt.3.34d0) then
                            if(yCenter(jj).gt.2.1d0) SoilType(ii,jj,kk)=3
                        endif
                        !Layer 3b
                        if(xCenter(ii).le.VSDownstream) then
                            if(yCenter(jj).gt.1.3d0.AND.yCenter(jj).lt.1.95d0) SoilType(ii,jj,kk)=2
                            !if(yCenter(jj).gt.1.3d0-0.2d0.AND.yCenter(jj).lt.1.95d0+0.2d0) SoilType(ii,jj,kk)=2    !FQ-Around conduit
                        endif
                        !Layer 3c
                        if(xCenter(ii).ge.VSUpstream) then
                            if(yCenter(jj).gt.1.2d0.AND.yCenter(jj).lt.2.1d0) SoilType(ii,jj,kk)=2
                            !if(yCenter(jj).gt.1.2d0-0.2d0.AND.yCenter(jj).lt.2.1d0+0.2d0) SoilType(ii,jj,kk)=2     !FQ-Around conduit
                        endif 
                    endif
                    !Layer 4
                    if(zCenter(kk).ge.0.75d0.AND.zCenter(kk).lt.1.0d0) then
                    !if(zCenter(kk).ge.0.75d0.AND.zCenter(kk).lt.1.05d0) then   !FQ-Around conduit
                        !Layer 4a
                        if(xCenter(ii).gt.1.15d0.AND.xCenter(ii).lt.VSDownstream) then
                            if(yCenter(jj).lt.1.3d0) SoilType(ii,jj,kk)=3
                        endif
                        !Layer 4b
                        if(xCenter(ii).le.VSDownstream) then
                            if(yCenter(jj).gt.1.3d0.AND.yCenter(jj).lt.1.95d0) SoilType(ii,jj,kk)=2
                            !if(yCenter(jj).gt.1.3d0-0.2d0.AND.yCenter(jj).lt.1.95d0+0.2d0) SoilType(ii,jj,kk)=2    !FQ-Around conduit
                        endif
                        !Layer 4c
                        if(xCenter(ii).ge.VSUpstream) then
                            if(yCenter(jj).gt.1.2d0.AND.yCenter(jj).lt.2.1d0) SoilType(ii,jj,kk)=2
                            !if(yCenter(jj).gt.1.2d0-0.2d0.AND.yCenter(jj).lt.2.1d0+0.2d0) SoilType(ii,jj,kk)=2     !FQ-Around conduit
                        endif 
                    endif
                    !Layer 5
                    if(zCenter(kk).ge.1.0d0.AND.zCenter(kk).lt.1.25d0) then
                    !if(zCenter(kk).ge.1.05d0.AND.zCenter(kk).lt.1.25d0) then   !FQ-Around conduit
                        !Layer 5a
                        if(xCenter(ii).gt.0.95d0.AND.xCenter(ii).lt.VSDownstream) then
                            if(yCenter(jj).gt.0.7d0.AND.yCenter(jj).lt.1.6d0) SoilType(ii,jj,kk)=3
                        endif
                        !Layer 5b
                        if(xCenter(ii).gt.VSUpstream.AND.xCenter(ii).lt.3.11d0) then
                            if(yCenter(jj).gt.0.7d0.AND.yCenter(jj).lt.1.6d0) SoilType(ii,jj,kk)=3
                        endif
                        !Layer 5c
                        if(xCenter(ii).gt.VSUpstream.AND.xCenter(ii).lt.2.51d0) then
                            if(yCenter(jj).lt.0.3d0) SoilType(ii,jj,kk)=2
                        endif 
                    endif
                    !Layer 6
                    if(zCenter(kk).ge.1.25d0.AND.zCenter(kk).lt.1.5d0) then
                        !Layer 6a
                        if(xCenter(ii).gt.VSUpstream.AND.xCenter(ii).lt.2.67d0) then
                            if(yCenter(jj).lt.0.64d0) SoilType(ii,jj,kk)=3
                        endif
                        !Layer 6b
                        if(xCenter(ii).gt.3.06d0.AND.xCenter(ii).lt.3.64d0) then
                            if(yCenter(jj).gt.0.77d0.AND.yCenter(jj).lt.1.39d0) SoilType(ii,jj,kk)=3
                        endif
                    endif
                    !Layer 7
                    if(zCenter(kk).ge.1.5d0.AND.zCenter(kk).lt.1.75d0) then
                        !Layer 7a
                        if(xCenter(ii).gt.1.44d0.AND.xCenter(ii).lt.VSDownstream) then
                            if(yCenter(jj).gt.1.74d0) SoilType(ii,jj,kk)=2
                        endif
                        !Layer 7b
                        if(xCenter(ii).gt.1.77d0.AND.xCenter(ii).lt.VSDownstream) then
                            if(yCenter(jj).gt.1.14d0.AND.yCenter(jj).lt.1.74d0) SoilType(ii,jj,kk)=2
                        endif
                        !Layer 7c
                        if(xCenter(ii).gt.1.14d0.AND.xCenter(ii).lt.1.77d0) then
                            if(yCenter(jj).gt.1.14d0.AND.yCenter(jj).lt.1.74d0) SoilType(ii,jj,kk)=3
                        endif
                        !Layer 7d
                        if(xCenter(ii).gt.2.84d0.AND.xCenter(ii).lt.3.51d0) then
                            if(yCenter(jj).gt.1.96d0) SoilType(ii,jj,kk)=2
                        endif 
                    endif!
                    !Verical screen
                    !if(xcp2(ii).gt.VSDownstream.AND.xcp1(ii).lt.VSUpstream) SoilType(ii,jj,kk)=2
                    if(xCenter(ii).gt.VSDownstream.AND.xCenter(ii).lt.VSUpstream) SoilType(ii,jj,kk)=2
                    !Epikarst
                    !if(zcp1(kk).ge.1.75d0) SoilType(ii,jj,kk)=3
                    if(zCenter(kk).ge.1.75d0) SoilType(ii,jj,kk)=3
                enddo
            enddo
        enddo
    
    end subroutine

    subroutine DefineSoilType_ORIGINAL
    !In this subroutine soil type is define for every CV.
    
!        integer (kind=4) ii,jj,kk
!    
!        !Set all CVs to principal material
!        SoilType=1       
!        
!!RETURN
!!****************************************************************!
!!
!!        !ONLY VERTICAL SCREEN
!!        do ii=-nExternal,nx-1+nExternal
!!            do jj=-nExternal,ny-1+nExternal
!!                do kk=-nExternal,nz-1+nExternal
!!                    !Verical screen
!!                    if(xCenter(ii).ge.1.95d0.AND.xCenter(ii).lt.2.21d0) SoilType(ii,jj,kk)=2
!!                enddo
!!            enddo
!!        enddo
!!
!!        !ONLY EPIKARST SCREEN
!!        do ii=-nExternal,nx-1+nExternal
!!            do jj=-nExternal,ny-1+nExternal
!!                do kk=-nExternal,nz-1+nExternal
!!                    !Epikarst
!!                    if(zCenter(kk).ge.1.75d0) SoilType(ii,jj,kk)=3
!!                enddo
!!            enddo
!!        enddo
!!    
!!RETURN
!!****************************************************************!
!
!
!        !Define zones of different conducivity
!        do ii=-nExternal,nx-1+nExternal
!            do jj=-nExternal,ny-1+nExternal
!                do kk=-nExternal,nz-1+nExternal
!                    !Layer 2
!                    if(zCenter(kk).ge.0.25d0.AND.zCenter(kk).lt.0.5d0) then
!                        !Layer 2a
!                        if(xCenter(ii).gt.0.73d0.AND.xCenter(ii).lt.1.33d0) then
!                            if(yCenter(jj).gt.0.43d0.AND.yCenter(jj).lt.1.07d0) SoilType(ii,jj,kk)=3
!                        endif
!                        !Layer 2b
!                        if(xCenter(ii).gt.2.83d0.AND.xCenter(ii).lt.3.46d0) then
!                            if(yCenter(jj).gt.0.28d0.AND.yCenter(jj).lt.0.87d0) SoilType(ii,jj,kk)=3
!                        endif                        
!                    endif
!                    !Layer 3
!                    if(zCenter(kk).ge.0.5d0.AND.zCenter(kk).lt.0.75d0) then
!                        !Layer 3a
!                        if(xCenter(ii).gt.2.21d0.AND.xCenter(ii).lt.3.34d0) then
!                            if(yCenter(jj).gt.2.1d0) SoilType(ii,jj,kk)=3
!                        endif
!                        !Layer 3b
!                        if(xCenter(ii).lt.1.95d0) then
!                            if(yCenter(jj).gt.1.3d0.AND.yCenter(jj).lt.1.95d0) SoilType(ii,jj,kk)=2
!                        endif
!                        !Layer 3c
!                        if(xCenter(ii).gt.2.21d0) then
!                            if(yCenter(jj).gt.1.2d0.AND.yCenter(jj).lt.2.1d0) SoilType(ii,jj,kk)=2
!                        endif 
!                    endif
!                    !Layer 4
!                    if(zCenter(kk).ge.0.75d0.AND.zCenter(kk).lt.1.0d0) then
!                        !Layer 4a
!                        if(xCenter(ii).gt.1.15d0.AND.xCenter(ii).lt.1.95d0) then
!                            if(yCenter(jj).lt.1.3d0) SoilType(ii,jj,kk)=3
!                        endif
!                        !Layer 4b
!                        if(xCenter(ii).lt.1.95d0) then
!                            if(yCenter(jj).gt.1.3d0.AND.yCenter(jj).lt.1.95d0) SoilType(ii,jj,kk)=2
!                        endif
!                        !Layer 4c
!                        if(xCenter(ii).gt.2.21d0) then
!                            if(yCenter(jj).gt.1.2d0.AND.yCenter(jj).lt.2.1d0) SoilType(ii,jj,kk)=2
!                        endif 
!                    endif
!                    !Layer 5
!                    if(zCenter(kk).ge.1.0d0.AND.zCenter(kk).lt.1.25d0) then
!                        !Layer 5a
!                        if(xCenter(ii).gt.0.95d0.AND.xCenter(ii).lt.1.95d0) then
!                            if(yCenter(jj).gt.0.7d0.AND.yCenter(jj).lt.1.6d0) SoilType(ii,jj,kk)=3
!                        endif
!                        !Layer 5b
!                        if(xCenter(ii).gt.2.21d0.AND.xCenter(ii).lt.3.11d0) then
!                            if(yCenter(jj).gt.0.7d0.AND.yCenter(jj).lt.1.6d0) SoilType(ii,jj,kk)=3
!                        endif
!                        !Layer 5c
!                        if(xCenter(ii).gt.2.21d0.AND.xCenter(ii).lt.2.51d0) then
!                            if(yCenter(jj).lt.0.3d0) SoilType(ii,jj,kk)=2
!                        endif 
!                    endif
!                    !Layer 6
!                    if(zCenter(kk).ge.1.25d0.AND.zCenter(kk).lt.1.5d0) then
!                        !Layer 6a
!                        if(xCenter(ii).gt.2.21d0.AND.xCenter(ii).lt.2.67d0) then
!                            if(yCenter(jj).lt.0.64d0) SoilType(ii,jj,kk)=3
!                        endif
!                        !Layer 6b
!                        if(xCenter(ii).gt.3.06d0.AND.xCenter(ii).lt.3.64d0) then
!                            if(yCenter(jj).gt.0.77d0.AND.yCenter(jj).lt.1.39d0) SoilType(ii,jj,kk)=3
!                        endif
!                    endif
!                    !Layer 7
!                    if(zCenter(kk).ge.1.5d0.AND.zCenter(kk).lt.1.75d0) then
!                        !Layer 7a
!                        if(xCenter(ii).gt.1.44d0.AND.xCenter(ii).lt.1.95d0) then
!                            if(yCenter(jj).gt.1.74d0) SoilType(ii,jj,kk)=2
!                        endif
!                        !Layer 7b
!                        if(xCenter(ii).gt.1.77d0.AND.xCenter(ii).lt.1.95d0) then
!                            if(yCenter(jj).gt.1.14d0.AND.yCenter(jj).lt.1.74d0) SoilType(ii,jj,kk)=2
!                        endif
!                        !Layer 7c
!                        if(xCenter(ii).gt.1.14d0.AND.xCenter(ii).lt.1.77d0) then
!                            if(yCenter(jj).gt.1.14d0.AND.yCenter(jj).lt.1.74d0) SoilType(ii,jj,kk)=3
!                        endif
!                        !Layer 7d
!                        if(xCenter(ii).gt.2.84d0.AND.xCenter(ii).lt.3.51d0) then
!                            if(yCenter(jj).gt.1.96d0) SoilType(ii,jj,kk)=2
!                        endif 
!                    endif!
!                    !Verical screen
!                    !if(xcp2(ii).gt.1.95d0.AND.xcp1(ii).lt.2.15d0) SoilType(ii,jj,kk)=2
!                    if(xCenter(ii).gt.1.95d0.AND.xCenter(ii).lt.2.21d0) SoilType(ii,jj,kk)=2
!                    !Epikarst
!                    !if(zcp1(kk).ge.1.75d0) SoilType(ii,jj,kk)=3
!                    if(zCenter(kk).ge.1.75d0) SoilType(ii,jj,kk)=3
!                enddo
!            enddo
!        enddo
    
    end subroutine
      
    subroutine InitialConditionTransform
    !FINITE VOLUME TRANSFORM OF INITIAL CONDITION.
    !This subroutine approximates initial condition (initial guess for steady-state solution).
    !Results are unknown coefficients at time t=0 needed as initial condition for time integration.

        integer (kind=4) ii,jj,kk,ic,jc,kc,icnt,jcnt,SimqKS
        integer (kind=4) icntNNZ
        integer (kind=4) ix,jY,kZ
        real (kind=8) coeff,c1,c2,c3

        !Set to zero
        icnt=0
        icntNNZ=1
        
        !PREPARE SYSTEM OF EQUATIONS
        !X direction
        do ii=-nExternal,nx-1+nExternal
            ix=idnint((xCenter(ii)-0.5d0*dx-dlx1)/(dx))
            !Y direction
            do jj=-nExternal,ny-1+nExternal
                jy=idnint((yCenter(jj)-0.5d0*dy-dly1)/(dy))
                !Z direction
                do kk=-nExternal,nz-1+nExternal
                    kz=idnint((zCenter(kk)-0.5d0*dz-dlz1)/(dz))
                    
                    !Equation counter
                    icnt=icnt+1
                    
                    if(LinSol.eq.'paradiso') IA(icnt)=icntNNZ

                    !Calculate non-zero coefficinets
                    do ic=iX-1,iX+1
                        do jc=jY-1,jY+1
                            do kc=kZ-1,kZ+1
 
                                !Coefficients column possition
                                jcnt=(kc+nExternal+1)+(jc-1+nExternal+1)*nzTOT+(ic-1+nExternal+1)*nzTOT*nyTOT

                                !Calculate matrix coefficient
                                coeff=BFIV(indx(ii),ic-iX)*DELX(indx(ii))*BFIV(indy(jj),jc-jY)*DELY(indy(jj))*BFIV(indz(kk),kc-kZ)*DELZ(indz(kk))
                                
                                !call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,c1)
                                !call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,c2)
                                !call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(kk),zcp2(kk),kmax,c3)
                                !coeff=c1*c2*c3
                                
                                !Put value in to matrix array
                                if(LinSol.eq.'simq') then
                                    ASIMQ(icnt,jcnt)=coeff
                                elseif(LinSol.eq.'paradiso') then
                                    JA(icntNNZ)=jcnt
                                    APARADISO(icntNNZ)=coeff
                                    icntNNZ=icntNNZ+sup1
                                endif
                                                            
                            enddo
                        enddo
                    enddo
                    
                    !RHS - Simplification: Int[f(x,y,z)]dxdydz=f(xc,yc,zc)*dx*dy*dz 
                    CC_t0(icnt)=IC_Matrix(xCenter(ii),yCenter(jj),zCenter(kk))*(xcp2(ii)-xcp1(ii))*(ycp2(jj)-ycp1(jj))*(zcp2(kk)-zcp1(kk))

                enddo
            enddo
        enddo
                       
        !!Write system of equations into data file
        !if(linsol.eq.'simq'.AND.writeSoE.eq.'yes') then
        !    open(lunF1,file='Ax=b_IC_Matrix.dat')
        !    do ii=1,NTOT3D
        !        do jj=1,NTOT3D    
        !            write(lunF1,'(e30.20,$)') ASIMQ(ii,jj)
        !        enddo
        !        write(lunF1,'(e30.20)') CC_t0(ii)
        !    enddo
        !    close(lunF1)
        !endif
    
        !Solve system of equation
        if(LinSol.eq.'simq') call Simq(ASIMQ,CC_t0,NTOT3D,SimqKS)
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) write(*,*) 'SIMQ singularity.'
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) read (*,*) 
        !
        if(LinSol.eq.'paradiso') IA(icnt+1)=NNZ+1
        if(LinSol.eq.'paradiso') call pardiso_sym_f90(NTOT3D,NNZ,IA,JA,APARADISO,CC_t0)

!=================================================================================!
        
        !Set matrix piezometric head solution from time "t" as initial guess for solution in time "t+dt3D"
        CC_LastIter=CC_t0
        
!=================================================================================!
!!!    !Set initial concentration matrix solution always to zero
        CCC_t0 = 0.0d0       
        CCC_LastIter=CCC_t0



    end subroutine
    
    
!!!  Solve advection step for ADE in 3-D porous matrix using backward particle tracking

       subroutine SolveAdvection3D(TimeE)
    
        
        
    !Results are unknown coefficients from advective step at time t+dt needed as initial condition 
    !for time integration of dispersion step at time t+dt. Higher order solution is obtained by using
    !the backward particle tracking. Finally, FCT stabilization is performed in oirder to eliminate
    !min-max oscillations.

        integer (kind=4) ii,jj,kk,ic,jc,kc,icnt,jcnt,SimqKS
        integer (kind=4) icntNNZ
        integer (kind=4) ix,jY,kZ
        real (kind=8) coeff,c1,c2,c3,m_i,u_i_max,u_i_min,f_ij
        real (kind=8) PIP(:),PIM(:),RIP(:),RIM(:),QIP(:),QIM(:)

        !Set to zero
        icnt=0
        icntNNZ=1
        
        TCurr=TimeCurr
        TEnd=TimeE
        dt3D=TEnd-TCurr !Single time step!        
        
        
        !PREPARE SYSTEM OF EQUATIONS
        !X direction
        do ii=-nExternal,nx-1+nExternal
            ix=idnint((xCenter(ii)-0.5d0*dx-dlx1)/(dx))
            !Y direction
            do jj=-nExternal,ny-1+nExternal
                jy=idnint((yCenter(jj)-0.5d0*dy-dly1)/(dy))
                !Z direction
                do kk=-nExternal,nz-1+nExternal
                    kz=idnint((zCenter(kk)-0.5d0*dz-dlz1)/(dz))
                    
                    !Equation counter
                    icnt=icnt+1
                    m_i = 0.0d0
                    
                    if(LinSol.eq.'paradiso') IA(icnt)=icntNNZ

                    !Calculate non-zero coefficinets
                    do ic=iX-1,iX+1
                        do jc=jY-1,jY+1
                            do kc=kZ-1,kZ+1
 
                                !Coefficients column possition
                                jcnt=(kc+nExternal+1)+(jc-1+nExternal+1)*nzTOT+(ic-1+nExternal+1)*nzTOT*nyTOT

                                !Calculate matrix coefficient
                                coeff=BFIV(indx(ii),ic-iX)*DELX(indx(ii))*BFIV(indy(jj),jc-jY)*DELY(indy(jj))*BFIV(indz(kk),kc-kZ)*DELZ(indz(kk))
                                            
                                !Put value in to matrix array
                                if(LinSol.eq.'simq') then
                                    ASIMQ(icnt,jcnt)=coeff
                                elseif(LinSol.eq.'paradiso') then
                                    JA(icntNNZ)=jcnt
                                    APARADISO(icntNNZ)=coeff
                                    icntNNZ=icntNNZ+sup1
                                endif
                                
                               m_i = m_i + coeff
                                                            
                            enddo
                        enddo
                    enddo
                    
                    !RHS 
                    CCC_t0(icnt)=Ptrack_Matrix(xCenter(ii),yCenter(jj),zCenter(kk))*DELX(indx(ii))*DELY(indy(jj))*DELZ(indz(kk))
                    CCC_t1(icnt)=CCC_t0(icnt)/m_i
                enddo
            enddo
        enddo
                       
        !!Write system of equations into data file
        !if(linsol.eq.'simq'.AND.writeSoE.eq.'yes') then
        !    open(lunF1,file='Ax=b_IC_Matrix.dat')
        !    do ii=1,NTOT3D
        !        do jj=1,NTOT3D    
        !            write(lunF1,'(e30.20,$)') ASIMQ(ii,jj)
        !        enddo
        !        write(lunF1,'(e30.20)') CC_t0(ii)
        !    enddo
        !    close(lunF1)
        !endif
    
        !Solve system of equations and get higher-order solution
        if(LinSol.eq.'simq') call Simq(ASIMQ,CCC_t0,NTOT3D,SimqKS)
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) write(*,*) 'SIMQ singularity.'
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) read (*,*) 
        !
        if(LinSol.eq.'paradiso') IA(icnt+1)=NNZ+1
        if(LinSol.eq.'paradiso') call pardiso_sym_f90(NTOT3D,NNZ,IA,JA,APARADISO,CCC_t0)


!!!  Perform FCT stabilization of advective solution in order to eliminate
!!!  sporious min-max oscillations

        icnt=0
        !PREPARE SYSTEM OF EQUATIONS
        !X direction
        do ii=-nExternal,nx-1+nExternal
            ix=idnint((xCenter(ii)-0.5d0*dx-dlx1)/(dx))
            !Y direction
            do jj=-nExternal,ny-1+nExternal
                jy=idnint((yCenter(jj)-0.5d0*dy-dly1)/(dy))
                !Z direction
                do kk=-nExternal,nz-1+nExternal
                    kz=idnint((zCenter(kk)-0.5d0*dz-dlz1)/(dz))
                    
                    !Equation counter
                    icnt=icnt+1
                    !Null all related fields for calculation
                    m_i=0.0d0
                    PIP(icnt) = 0.0d0
                    PIM(icnt) = 0.0d0
                    u_i_max = -1.d9
                    u_i_min =  1.d9
                    
                    !Calculate non-zero coefficinets
                    do ic=iX-1,iX+1
                        do jc=jY-1,jY+1
                            do kc=kZ-1,kZ+1
 
                                !Coefficients column possition
                                jcnt=(kc+nExternal+1)+(jc-1+nExternal+1)*nzTOT+(ic-1+nExternal+1)*nzTOT*nyTOT

                                !Calculate matrix coefficient
                                coeff=BFIV(indx(ii),ic-iX)*DELX(indx(ii))*BFIV(indy(jj),jc-jY)*DELY(indy(jj))*BFIV(indz(kk),kc-kZ)*DELZ(indz(kk))
                                m_i = m_i + coeff            
                                coeff = coeff * (CCC_t0(icnt)-CCC_t0(jcnt))
                               
                                if (icnt.ne.jcnt) PIP(icnt) = PIP(icnt) + dmax1(0.0d0, coeff)
                                if (icnt.ne.jcnt) PIM(icnt) = PIN(icnt) + dmin1(0.0d0, coeff)
                                if (icnt.ne.jcnt) u_i_max = dmax1(u_i_max,CCC_t0(jcnt))
                                if (icnt.ne.jcnt) u_i_min = dmin1(u_i_min,CCC_t0(jcnt))                            
                            enddo
                        enddo
                    enddo
                    
                    !FCT Stabilization calculation 
                    QIP(icnt) = m_i*(u_i_max-CCC_t1(icnt))
                    QIM(icnt) = m_i*(u_i_min-CCC_t1(icnt))
                    RIP(icnt) = min(1.0d0,QIP(icnt)/PIP(icnt))
                    RIM(icnt) = min(1.0d0,QIM(icnt)/PIM(icnt))
                    
                enddo
            enddo
        enddo
                       
        icnt=0
        !PREPARE SYSTEM OF EQUATIONS
        !X direction
        do ii=-nExternal,nx-1+nExternal
            ix=idnint((xCenter(ii)-0.5d0*dx-dlx1)/(dx))
            !Y direction
            do jj=-nExternal,ny-1+nExternal
                jy=idnint((yCenter(jj)-0.5d0*dy-dly1)/(dy))
                !Z direction
                do kk=-nExternal,nz-1+nExternal
                    kz=idnint((zCenter(kk)-0.5d0*dz-dlz1)/(dz))
                    
                    !Equation counter
                    icnt=icnt+1
                    !Null all related fields for calculation
                    m_i = 0.0d0
                    f_ij = 0.0d0
                    
                    !Calculate non-zero coefficinets
                    do ic=iX-1,iX+1
                        do jc=jY-1,jY+1
                            do kc=kZ-1,kZ+1
 
                                !Coefficients column possition
                                jcnt=(kc+nExternal+1)+(jc-1+nExternal+1)*nzTOT+(ic-1+nExternal+1)*nzTOT*nyTOT

                                !Calculate matrix coefficient
                                coeff=BFIV(indx(ii),ic-iX)*DELX(indx(ii))*BFIV(indy(jj),jc-jY)*DELY(indy(jj))*BFIV(indz(kk),kc-kZ)*DELZ(indz(kk))
                                m_i = m_i + coeff            
                                coeff = coeff * (CCC_t0(icnt)-CCC_t0(jcnt))
                                
                                if (coeff.gt.0.0d0) coeff = coeff*dmin1(RIP(icnt),RIM(jcnt))
                                if (coeff.le.0.0d0) coeff = coeff*dmin1(RIM(icnt),RIP(jcnt))                                                            
                                if (icnt.ne.jcnt) f_ij = f_ij + coeff
                            enddo
                        enddo
                    enddo
                    
                    !FCT Stabilization advective solution 
                    CCC_t1(icnt) = CCC_t1(icnt) + f_ij/m_i
                    
                enddo
            enddo
        enddo
                       





!=================================================================================!
        
        !Set matrix solution from time "t" as initial guess for solution in time "t+dt3D"
        CCC_LastIter=CCC_t1
        
!=================================================================================!
 

    end subroutine
    
     
        
!!!  Solve dispersion step for ADE in 3-D porous matrix using CV-IGA with dispersion,
!!!  time, boundary and source-exchange terms applying initial conditions from advective step

       subroutine SolveDispersion3D(TimeE)
    
        
        
    !Results are unknown coefficient for time integration of dispersion step at time t+dt.

        integer (kind=4) ii,jj,kk,ic,jc,kc,icnt,jcnt,SimqKS
        integer (kind=4) icntNNZ
        integer (kind=4) ix,jY,kZ
        real (kind=8) coeff,coeff1,coeff2,coeff3,c1,c2,c3,TimeE
        integer (kind=4) icnteps,CoorEps(3)
        logical          Recharge_Exist,Pumping_Exist
        real*8           Recharge_rate,Conc_rate,Pumping_rate

        !Set to zero
        icnt=0
        icntNNZ=1
        
        TCurr=TimeCurr
        TEnd=TimeE
        dt3D=TEnd-TCurr !Single time step!        
        
        
        !PREPARE SYSTEM OF EQUATIONS
        !X direction
        do ii=-nExternal,nx-1+nExternal
            ix=idnint((xCenter(ii)-0.5d0*dx-dlx1)/(dx))
            !Y direction
            do jj=-nExternal,ny-1+nExternal
                jy=idnint((yCenter(jj)-0.5d0*dy-dly1)/(dy))
                !Z direction
                do kk=-nExternal,nz-1+nExternal
                    kz=idnint((zCenter(kk)-0.5d0*dz-dlz1)/(dz))
                    
                    !Equation counter
                    icnt=icnt+1
                    
                    if(LinSol.eq.'paradiso') IA(icnt)=icntNNZ

                    !Calculate non-zero coefficinets
                    do ic=iX-1,iX+1
                        do jc=jY-1,jY+1
                            do kc=kZ-1,kZ+1
 
                                !Coefficients column possition
                                jcnt=(kc+nExternal+1)+(jc-1+nExternal+1)*nzTOT+(ic-1+nExternal+1)*nzTOT*nyTOT

                                !Calculate matrix coefficient
                                
                                
                                !Add firstly time capacity term (so called mass-matrix)
                
                                     
                                coeff=BFIV(indx(ii),ic-iX)*DELX(indx(ii))*BFIV(indy(jj),jc-jY)*DELY(indy(jj))*BFIV(indz(kk),kc-kZ)*DELZ(indz(kk))
                                coeff = coeff / dt3D    
                                
                    !Add dispersive term at all six control surfaces of 
                    !current control volume (except in Neumman boundaries)
  
                    if (dabs(indx(ii)).ne.2.and.dabs(indy(jj)).ne.2.and.dabs(indz(kk)).ne.2) then
                    
 !!!    Surface xcp2(ii)
 
        coeff1 = bfun(nOrder,xVertex(ic),xcp2(ii),dx,1,dlx1,dlx2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff - D_ij(1,1,xcp2(ii),yCenter(jj),zCenter(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,xVertex(ic),xcp2(ii),dx,0,dlx1,dlx2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,1,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff - D_ij(1,2,xcp2(ii),yCenter(jj),zCenter(kk))*coeff1*coeff2*coeff3
                    
        coeff1 = bfun(nOrder,xVertex(ic),xcp2(ii),dx,0,dlx1,dlx2)
        call trap_int_1D(bfun,nOrder,yCenter(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zCenter(kc),dz,1,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff - D_ij(1,3,xcp2(ii),yCenter(jj),zCenter(kk))*coeff1*coeff2*coeff3
        
!!!    Surface xcp1(ii)
 
        coeff1 = bfun(nOrder,xVertex(ic),xcp1(ii),dx,1,dlx1,dlx2)
        call trap_int_1D( bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + D_ij(1,1,xcp1(ii),yCenter(jj),zCenter(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,xVertex(ic),xcp1(ii),dx,0,dlx1,dlx2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,1,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + D_ij(1,2,xcp1(ii),yCenter(jj),zCenter(kk))*coeff1*coeff2*coeff3
                    
        coeff1 = bfun(nOrder,xVertex(ic),xcp1(ii),dx,0,dlx1,dlx2)
        call trap_int_1D(bfun,nOrder,yCenter(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zCenter(kc),dz,1,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + D_ij(1,3,xcp1(ii),yCenter(jj),zCenter(kk))*coeff1*coeff2*coeff3

 !!!    Surface ycp2(ii)
 
        coeff1 = bfun(nOrder,yVertex(jc),ycp2(jj),dy,1,dly1,dly2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff - D_ij(2,2,xCenter(ii),ycp2(jj),zCenter(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,yVertex(jc),ycp2(jj),dy,0,dly1,dly2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,1,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff - D_ij(2,1,xCenter(ii),ycp2(jj),zCenter(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,yVertex(jc),ycp2(jj),dy,0,dly1,dly2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,1,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff - D_ij(2,3,xCenter(ii),ycp2(jj),zCenter(kk))*coeff1*coeff2*coeff3

!!!    Surface ycp1(ii)
 
        coeff1 = bfun(nOrder,yVertex(jc),ycp1(jj),dy,1,dly1,dly2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + D_ij(2,2,xCenter(ii),ycp1(jj),zCenter(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,yVertex(jc),ycp1(jj),dy,0,dly1,dly2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,1,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + D_ij(2,1,xCenter(ii),ycp1(jj),zCenter(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,yVertex(jc),ycp1(jj),dy,0,dly1,dly2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,1,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + D_ij(2,3,xCenter(ii),ycp1(jj),zCenter(kk))*coeff1*coeff2*coeff3

!!!    Surface zcp2(ii)
 
        coeff1 = bfun(nOrder,zVertex(kc),zcp2(jj),dz,1,dlz1,dlz2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff3)
        coeff = coeff - D_ij(3,3,xCenter(ii),yCenter(jj),zcp2(kk))*coeff1*coeff2*coeff3

 
        coeff1 = bfun(nOrder,zVertex(kc),zcp2(jj),dz,0,dlz1,dlz2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,1,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff3)
        coeff = coeff - D_ij(3,1,xCenter(ii),yCenter(jj),zcp2(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,zVertex(kc),zcp2(jj),dz,0,dlz1,dlz2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,1,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff3)
        coeff = coeff - D_ij(3,2,xCenter(ii),yCenter(jj),zcp2(kk))*coeff1*coeff2*coeff3

!!!    Surface zcp1(ii)
 
        coeff1 = bfun(nOrder,zVertex(kc),zcp1(jj),dz,1,dlz1,dlz2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff3)
        coeff = coeff - D_ij(3,3,xCenter(ii),yCenter(jj),zcp1(kk))*coeff1*coeff2*coeff3

 
        coeff1 = bfun(nOrder,zVertex(kc),zcp1(jj),dz,0,dlz1,dlz2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,1,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff3)
        coeff = coeff - D_ij(3,1,xCenter(ii),yCenter(jj),zcp1(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,zVertex(kc),zcp1(jj),dz,0,dlz1,dlz2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,1,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff3)
        coeff = coeff - D_ij(3,2,xCenter(ii),yCenter(jj),zcp1(kk))*coeff1*coeff2*coeff3

         !Add dispersive term and Dirichlet penalization term for DBC in Dirichlet boundaries
 
        else if (indx(ii).eq.-2.and.BCType_Transport(1).eq.'Dirichlet') then 
                    
!!!    Surface xcp1(ii)
 
!!!!!!!!!!!!!!!!!   DBC
        coeff1 = bfun(nOrder,xVertex(ic),xcp1(ii),dx,0,dlx1,dlx2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + coeff1*coeff2*coeff3
!!!!!!!!!!!!!!!!!   DBC

        coeff1 = bfun(nOrder,xVertex(ic),xcp1(ii),dx,1,dlx1,dlx2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + D_ij(1,1,xcp1(ii),yCenter(jj),zCenter(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,xVertex(ic),xcp1(ii),dx,0,dlx1,dlx2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,1,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + D_ij(1,2,xcp1(ii),yCenter(jj),zCenter(kk))*coeff1*coeff2*coeff3
                    
        coeff1 = bfun(nOrder,xVertex(ic),xcp1(ii),dx,0,dlx1,dlx2)
        call trap_int_1D(bfun,nOrder,yCenter(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zCenter(kc),dz,1,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + D_ij(1,3,xcp1(ii),yCenter(jj),zCenter(kk))*coeff1*coeff2*coeff3

        
                    else if (indx(ii).eq. 2.and.BCType_Transport(2).eq.'Dirichlet') then 
                    
 !!!    Surface xcp2(ii)
 
!!!!!!!!!!!!!!!!!   DBC
        coeff1 = bfun(nOrder,xVertex(ic),xcp2(ii),dx,0,dlx1,dlx2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + coeff1*coeff2*coeff3
!!!!!!!!!!!!!!!!!   DBC

        coeff1 = bfun(nOrder,xVertex(ic),xcp2(ii),dx,1,dlx1,dlx2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff - D_ij*(1,1,xcp2(ii),yCenter(jj),zCenter(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,xVertex(ic),xcp2(ii),dx,0,dlx1,dlx2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,1,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff - D_ij*(1,2,xcp2(ii),yCenter(jj),zCenter(kk))*coeff1*coeff2*coeff3
                    
        coeff1 = bfun(nOrder,xVertex(ic),xcp2(ii),dx,0,dlx1,dlx2)
        call trap_int_1D(bfun,nOrder,yCenter(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zCenter(kc),dz,1,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff - D_ij*(1,3,xcp2(ii),yCenter(jj),zCenter(kk))*coeff1*coeff2*coeff3
        
         else if (indy(jj).eq.-2.and.BCType_Transport(3).eq.'Dirichlet') then 

!!!    Surface ycp1(ii)
 
!!!!!!!!!!!  DBC
        coeff1 = bfun(nOrder,yVertex(jc),ycp1(jj),dy,0,dly1,dly2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + coeff1*coeff2*coeff3
!!!!!!!!!!!  DBC

        coeff1 = bfun(nOrder,yVertex(jc),ycp1(jj),dy,1,dly1,dly2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + D_ij*(2,2,xCenter(ii),ycp1(jj),zCenter(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,yVertex(jc),ycp1(jj),dy,0,dly1,dly2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,1,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + D_ij*(2,1,xCenter(ii),ycp1(jj),zCenter(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,yVertex(jc),ycp1(jj),dy,0,dly1,dly2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,1,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + D_ij*(2,3,xCenter(ii),ycp1(jj),zCenter(kk))*coeff1*coeff2*coeff3

          else if (indy(jj).eq. 2.and.BCType_Transport(4).eq.'Dirichlet') then 

!!!    Surface ycp2(ii)
 
!!!!!!!!!!!  DBC
        coeff1 = bfun(nOrder,yVertex(jc),ycp2(jj),dy,0,dly1,dly2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff + coeff1*coeff2*coeff3
!!!!!!!!!!!  DBC

        coeff1 = bfun(nOrder,yVertex(jc),ycp2(jj),dy,1,dly1,dly2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff - D_ij*(2,2,xCenter(ii),ycp2(jj),zCenter(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,yVertex(jc),ycp2(jj),dy,0,dly1,dly2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,1,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,0,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff - D_ij*(2,1,xCenter(ii),ycp2(jj),zCenter(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,yVertex(jc),ycp2(jj),dy,0,dly1,dly2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,zVertex(kc),dz,1,dlz1,dlz2,zcp1(jj),zcp2(jj),kmax,coeff3)
        coeff = coeff - D_ij*(2,3,xCenter(ii),ycp2(jj),zCenter(kk))*coeff1*coeff2*coeff3

         else if (indz(kk).eq.-2.and.BCType_Transport(5).eq.'Dirichlet') then 

 !!!    Surface zcp1(ii)
 
!!!!!!!!!!   DBC
        coeff1 = bfun(nOrder,zVertex(kc),zcp1(jj),dz,0,dlz1,dlz2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff3)
        coeff = coeff + coeff1*coeff2*coeff3
!!!!!!!!!!   DBC
 
        coeff1 = bfun(nOrder,zVertex(kc),zcp1(jj),dz,1,dlz1,dlz2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff3)
        coeff = coeff - D_ij*(3,3,xCenter(ii),yCenter(jj),zcp1(kk))*coeff1*coeff2*coeff3

 
        coeff1 = bfun(nOrder,zVertex(kc),zcp1(jj),dz,0,dlz1,dlz2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,1,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff3)
        coeff = coeff - D_ij*(3,1,xCenter(ii),yCenter(jj),zcp1(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,zVertex(kc),zcp1(jj),dz,0,dlz1,dlz2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,1,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff3)
        coeff = coeff - D_ij*(3,2,xCenter(ii),yCenter(jj),zcp1(kk))*coeff1*coeff2*coeff3

         else if (indz(kk).eq. 2.and.BCType_Transport(6).eq.'Dirichlet') then 

!!!    Surface zcp2(ii)
 
!!!!!!!!!!   DBC
        coeff1 = bfun(nOrder,zVertex(kc),zcp2(jj),dz,0,dlz1,dlz2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff3)
        coeff = coeff + coeff1*coeff2*coeff3
!!!!!!!!!!   DBC

        coeff1 = bfun(nOrder,zVertex(kc),zcp2(jj),dz,1,dlz1,dlz2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff3)
        coeff = coeff - D_ij*(3,3,xCenter(ii),yCenter(jj),zcp2(kk))*coeff1*coeff2*coeff3

 
        coeff1 = bfun(nOrder,zVertex(kc),zcp2(jj),dz,0,dlz1,dlz2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,1,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,0,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff3)
        coeff = coeff - D_ij*(3,1,xCenter(ii),yCenter(jj),zcp2(kk))*coeff1*coeff2*coeff3

        coeff1 = bfun(nOrder,zVertex(kc),zcp2(jj),dz,0,dlz1,dlz2)
        call trap_int_1D(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff2)
        call trap_int_1D(bfun,nOrder,yVertex(jc),dy,1,dly1,dly2,ycp1(jj),ycp2(jj),kmax,coeff3)
        coeff = coeff - D_ij*(3,2,xCenter(ii),yCenter(jj),zcp2(kk))*coeff1*coeff2*coeff3


          end if


             !Add exchange advective flux if exchange flow is oriented from matrix toward the conduit
             !C_ex=Conc_3D
             
                   ConduitEx=.false.
                    if(xcp1(ii).lt.x2Conduit.AND.xcp2(ii).gt.x1Conduit) then
                        if(ycp1(jj).lt.yConduit.AND.ycp2(jj).gt.yConduit) then
                            if(zcp1(kk).lt.zConduit.AND.zcp2(kk).gt.zConduit) ConduitEx=.true.
                        endif
                    endif
      
  
                
                coeff = coeff + ExchangeAdvectiveflux_M_C_BFUN (xcp1(ii),xcp2(ii),ycp1(jj),ycp2(jj),zcp1(kk),zcp2(kk),ConduitEx) 
                
             !Add loading term that comes from the pumping
             
               call Pumping_Transport(ii,jj,kk,Pumping_Exist,Pumping_rate)
               
               If (Pumping_Exist.eq..true.) then
               coeff1 = bfun(nOrder,xVertex(ic),xCenter(ic),dx,0,dlx1,dlx2)
               coeff2 = bfun(nOrder,yVertex(jc),yCenter(jc),dy,0,dly1,dly2)
               coeff3 = bfun(nOrder,zVertex(kc),zCenter(kc),dz,0,dlz1,dlz2)
               coeff = coeff - Pumping_rate*coeff1*coeff2*coeff3
               end if
                
                
                                
                                !Put value in to matrix array
                                if(LinSol.eq.'simq') then
                                    ASIMQ(icnt,jcnt)=coeff
                                elseif(LinSol.eq.'paradiso') then
                                    JA(icntNNZ)=jcnt
                                    APARADISO(icntNNZ)=coeff
                                    icntNNZ=icntNNZ+sup1
                                endif
                                                            
                            enddo
                        enddo
                    enddo
                    
                    !RHS 

             !Add exchange advective flux if exchange flow is oriented from the conduit toward the matrix
             !C_ex=Conc_1D
              
CCC_t1(icnt)=ExchangeAdvectiveflux_M_C_Conc(xcp1(ii),xcp2(ii),ycp1(jj),ycp2(jj),zcp1(kk),zcp2(kk),ConduitEx) 

            !Influence of time term from the start of the current time step (indeed advective concentration)
            
!            call trap_int_1D(Conc_LastStep_Conduit,xcp1(ii),xcp2(ii),kmax,delta_CCC)
            delta_CCC = Conc_3D(xCenter(ii),yCenter(jj),zCenter(kk))*DELX(indx(ii))*DELY(indy(jj))*DELZ(indz(kk))
            CCC_t1(icnt) = CCC_t1(icnt) + delta_CCC / dt3D
            
            !Influence of Neumann boundary conditions or Dirichlet boundary conditions (penalization term)
            
            if (indx(ii).eq.-2) then
            CCC_t1(icnt) = CCC_t1(icnt) + BCValue_Transport(1)*DELY(indy(jj))*DELZ(indz(kk))
            end if

            if (indx(ii).eq. 2) then
            CCC_t1(icnt) = CCC_t1(icnt) + BCValue_Transport(2)*DELY(indy(jj))*DELZ(indz(kk))
            end if

            if (indy(jj).eq.-2) then
            CCC_t1(icnt) = CCC_t1(icnt) + BCValue_Transport(3)*DELX(indx(ii))*DELZ(indz(kk))
            end if

            if (indy(jj).eq. 2) then
            CCC_t1(icnt) = CCC_t1(icnt) + BCValue_Transport(4)*DELX(indx(ii))*DELZ(indz(kk))
            end if

            if (indz(kk).eq.-2) then
            CCC_t1(icnt) = CCC_t1(icnt) + BCValue_Transport(5)*DELX(indx(ii))*DELY(indy(jj))
            end if

            if (indz(kk).eq. 2) then
            CCC_t1(icnt) = CCC_t1(icnt) + BCValue_Transport(6)*DELX(indx(ii))*DELY(indy(jj))
            end if
            
             !Add loading term that comes from the recharge with specified fluid concentration
             
               call Recharge_Transport(ii,jj,kk,Recharge_Exist,Recharge_rate,Conc_rate)
               
               If (Recharge_Exist.eq..true.) then
               CCC_t1(icnt) = CCC_t1(icnt) + Recharge_rate*Conc_rate
                              end if
                
                
            

                enddo
            enddo
        enddo
                       
        !!Write system of equations into data file
        !if(linsol.eq.'simq'.AND.writeSoE.eq.'yes') then
        !    open(lunF1,file='Ax=b_IC_Matrix.dat')
        !    do ii=1,NTOT3D
        !        do jj=1,NTOT3D    
        !            write(lunF1,'(e30.20,$)') ASIMQ(ii,jj)
        !        enddo
        !        write(lunF1,'(e30.20)') CC_t0(ii)
        !    enddo
        !    close(lunF1)
        !endif
    
        !Solve system of equation
        if(LinSol.eq.'simq') call Simq(ASIMQ,CCC_t1,NTOT3D,SimqKS)
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) write(*,*) 'SIMQ singularity.'
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) read (*,*) 
        !
        if(LinSol.eq.'paradiso') IA(icnt+1)=NNZ+1
        if(LinSol.eq.'paradiso') call pardiso_sym_f90(NTOT3D,NNZ,IA,JA,APARADISO,CCC_t1)
		
		
           call CheckConvergence_Transport(epsCurrTransport,IcntEps,CoorEps)
!=================================================================================!
        
        CCC_LastIter=CCC_t1
        
!=================================================================================!
 

    end subroutine
    
    
!!!   Subroutine defines recharge and concentration rates for specified number of recharge points
!!!   Each point will belong to one control volume
!!!   And for Recharge surfaces for rain and top surface injection
    
     subroutine Recharge_Transport(ir,jr,kr,Recharge_Exist,Recharge_rate,Conc_rate,Time_R)

     integer*4   ir,jr,kr,ireach,ip,jp,kp
     real*8      Recharge_Exist,Recharge_rate,Conc_rate,Time_R
     logical     Recharge_Exist
     
     Recharge_Exist = .false.
     
     do ireach = 1, Recharge_points
     ip=idnint((xRecharge(ireach)-0.5d0*dx-dlx1)/(dx))
     jp=idnint((yRecharge(ireach)-0.5d0*dy-dly1)/(dy))
     kp=idnint((zRecharge(ireach)-0.5d0*dz-dlz1)/(dz))
     
     if (ip.eq.ir.and.jp.eq.jr.and.kp.eq.kr) then
     Recharge_Exist = .true.
     Recharge_rate = Q_Recharge(ireach,Time_R)
     Conc_rate = C_Recharge(ireach,Time_R)
     return
     end if
     
     end do

     do ireach = 1, Recharge_surfaces
     ip1=idnint((x1Recharge(ireach)-0.5d0*dx-dlx1)/(dx))
     jp1=idnint((y1Recharge(ireach)-0.5d0*dy-dly1)/(dy))
     ip2=idnint((x2Recharge(ireach)-0.5d0*dx-dlx1)/(dx))
     jp2=idnint((y2Recharge(ireach)-0.5d0*dy-dly1)/(dy))
     kp=idnint((zRecharge(ireach)-0.5d0*dz-dlz1)/(dz))
     
     if (ip.ge.ip1.and.ip.le.ip2.and.jp.ge.jp1.and.jp.le.jp2) then
     Recharge_Exist = .true.
     Recharge_rate = Q_Recharge(ireach+Recharge_points,Time_R)
     Conc_rate = C_Recharge(ireach+Recharge_points,Time_R)
     return
     end if
     
     end do

        
     end subroutine

!!!   Subroutine defines pumping rates for specified number of pumping points
!!!   - each point will belong to one control volume
    
     subroutine Pumping_Transport(ir,jr,kr,Pumping_Exist,Pumping_rate,Time_R)

     integer*4   ir,jr,kr,ipump,ip,jp,kp
     real*8      Pumping_rate,Time_R
     logical     Pumping_Exist
     
     Pumping_Exist = .false.
     
     do ipump = 1, Pumping_points
     ip=idnint((xPump(ipump)-0.5d0*dx-dlx1)/(dx))
     jp=idnint((yPump(ipump)-0.5d0*dy-dly1)/(dy))
     kp=idnint((zPump(ipump)-0.5d0*dz-dlz1)/(dz))
     
     if (ip.eq.ir.and.jp.eq.jr.and.kp.eq.kr) then
     Pumping_Exist = .true.
     Pumping_rate = Q_Pump(ipump,Time_R)
     return
     end if
     
     end do
        
     end subroutine

     real*8 function Q_Pump(ipump,Time_R)
     integer*4 ipump
     real*8    Time_R 
     
     if (ipump.eq.1) then
     Q_Pump = 0.0d0
     if (Time_R.le.10000.0d0.and.Time_R.ge.5000.0d0) Q_Pump = 1.0d0
     end if
     
     end function   
    
     real*8 function Q_Recharge(ireach,Time_R)
     integer*4 ireach
     real*8    Time_R 
     
     if (ireach.eq.1) then
     Q_Recharge = 0.0d0
     if (Time_R.le.10000.0d0.and.Time_R.ge.5000.0d0) Q_Recharge = 1.0d0
     end if
     
     end function   
    
     real*8 function C_Recharge(ireach,Time_R)
     integer*4 ireach
     real*8    Time_R 
     
     if (ireach.eq.1) then
     C_Recharge = 0.0d0
     if (Time_R.le.10000.0d0.and.Time_R.ge.5000.0d0) C_Recharge = 1.0d0
     end if
     
     end function   
    
!!!  Function which calculates dispersion tensor in space point (xp,yp,zp)
!!!  for velocity field at t+dt

     real*8  function D_ij(idir,jdir,xp,yp,zp)
     
     integer*4 idir,jdir
     real*8    xp,yp,zp,vp,vpi,vpj,vpx,vpy,vpz
     integer (kind=8) ip,jp,kp,isoil
        
        
          ip=idnint((xp-0.5d0*dx-dlx1)/(dx))
          jp=idnint((yp-0.5d0*dy-dly1)/(dy))                
          kp=idnint((zp-0.5d0*dz-dlz1)/(dz))          
          
        isoil = SoilType(ip,jp,kp)
        Por = Porosity(isoil)
     
     vpx = Vm3D(xp,yp,zp,1,0,0)
     vpy = Vm3D(xp,yp,zp,0,1,0)
     vpz = Vm3D(xp,yp,zp,0,0,1)
     vp=dsqrt(vpx**2+vpy**2+vpz**2)
     
        if (idir.eq.1) vpi = vpx
        if (idir.eq.2) vpi = vpy
        if (idir.eq.3) vpi = vpz
     
        if (jdir.eq.1) vpj = vpx
        if (jdir.eq.2) vpj = vpy
        if (jdir.eq.3) vpj = vpz
        
        D_ij = Diffusion_matrix*Delta_Dirac(idir,jdir) +       &
               alfa_t*Delta_Dirac(idir,jdir)*vp
        if (dabs(vp).gt.1.d-13) D_ij=D_ij+((alfa_l-alfa_t)*vpi*vpj)/vp
      
     
     end function D_ij   
     
       real*8  function Delta_Dirac(idir,jdir) 
       integer*4   idir,jdir
       
       Delta_Dirac = 0.0d0
       
       if (idir.eq.jdir) Delta_Dirac = 1.0d0
       
       end function Delta_Dirac
       
                
     
    subroutine AssembleMatrixEquations
    !Assemble system of discretized equations. Coefficinets are calculated by using latest Picard iteration values&
    !and by averaging (one point Gauss integration) over each CV.
    
        integer (kind=4) ii,jj,kk,ic,jc,kc,icnt,jcnt
        integer (kind=4) ix,jY,kZ,mdx,mdy,mdz,IT
        integer (kind=4) icntNNZ
        real (kind=8) CVvolume,CVdx,CVdy,CVdz,CenterCVbfunValue
        real (kind=8) Kije,Kijw,Kijn,Kijs,Kijt,Kijb
        real (kind=8) KijDBCe,KijDBCw,KijDBCn,KijDBCs,KijDBCt,KijDBCb
        real (kind=8) BFUNe,BFUNw,BFUNn,BFUNs,BFUNt,BFUNb
        real (kind=8) Se_C,SeModes_C(2),SIDbcValue
        real (kind=8) Kij,Sij,Cij,OOWi
        real (kind=8) NUR
        real (kind=8) coeff
        logical ConduitEx,DirichletBC
        
        !Set to zero
        icnt=0
        icntNNZ=1
        CC_t1=0.d0
        if(LinSol.eq.'simq') ASIMQ=0.d0
        if(LinSol.eq.'paradiso') APARADISO=0.d0
        
        !PREPARE SYSTEM OF EQUATIONS
        !X direction
        DO ii=-nExternal,nx-1+nExternal
            ix=idnint((xCenter(ii)-0.5d0*dx-dlx1)/(dx))
            CVdx=(xcp2(ii)-xcp1(ii))
            !Y direction
            DO jj=-nExternal,ny-1+nExternal
                jy=idnint((yCenter(jj)-0.5d0*dy-dly1)/(dy))
                CVdy=(ycp2(jj)-ycp1(jj))
                !Z direction
                DO kk=-nExternal,nz-1+nExternal
                    kz=idnint((zCenter(kk)-0.5d0*dz-dlz1)/(dz))
                    CVdz=(zcp2(kk)-zcp1(kk))
                    
                    !Equation counter
                    icnt=icnt+1
                    
                    !Nonzero array                    
                    if(LinSol.eq.'paradiso') IA(icnt)=icntNNZ
                    
                    !Check for pipe in current CV - for karst flow model
                    ConduitEx=.false.
                    if(xcp1(ii).lt.x2Conduit.AND.xcp2(ii).gt.x1Conduit) then
                        if(ycp1(jj).lt.yConduit.AND.ycp2(jj).gt.yConduit) then
                            if(zcp1(kk).lt.zConduit.AND.zcp2(kk).gt.zConduit) ConduitEx=.true.
                        endif
                    endif
                    
                    !Volume of current CV 
                    CVvolume=CVdx*CVdy*CVdz
                    
                    !CV's average value of effective saturation
                    IT=SoilType(ii,jj,kk)
                    Se_C=EffectiveSaturation(IT,CXYZ(xCenter(ii),yCenter(jj),zCenter(kk),0,0,0,CC_LastIter)-zCenter(kk),SeModes_C)
                    
                    !Prepare part of conductivity-matrix contribution
                    call ConductivityMatrixContribution(ii,jj,kk,CVdx,CVdy,CVdz,Kije,Kijw,Kijn,Kijs,Kijt,Kijb)
                    
                    !Set to zero
                    KijDBCe=0.d0; KijDBCw=0.d0; KijDBCn=0.d0; KijDBCs=0.d0; KijDBCt=0.d0; KijDBCb=0.d0; SIDbcValue=0.d0
                    
                    !Boundary condition modification
                    DirichletBC=.false.
                    if(indx(ii).eq.+2) call BoundaryConditionsModifications(1,ii,jj,kk,CVdy*CVdz,Kije,KijDBCe,CC_t1(icnt),DirichletBC,SIDbcValue)
                    if(indx(ii).eq.-2) call BoundaryConditionsModifications(2,ii,jj,kk,CVdy*CVdz,Kijw,KijDBCw,CC_t1(icnt),DirichletBC,SIDbcValue)
                    if(indy(jj).eq.+2) call BoundaryConditionsModifications(3,ii,jj,kk,CVdx*CVdz,Kijn,KijDBCn,CC_t1(icnt),DirichletBC,SIDbcValue)
                    if(indy(jj).eq.-2) call BoundaryConditionsModifications(4,ii,jj,kk,CVdx*CVdz,Kijs,KijDBCs,CC_t1(icnt),DirichletBC,SIDbcValue)
                    if(indz(kk).eq.+2) call BoundaryConditionsModifications(5,ii,jj,kk,CVdx*CVdy,Kijt,KijDBCt,CC_t1(icnt),DirichletBC,SIDbcValue)
                    if(indz(kk).eq.-2) call BoundaryConditionsModifications(6,ii,jj,kk,CVdx*CVdy,Kijb,KijDBCb,CC_t1(icnt),DirichletBC,SIDbcValue)

                    !Calculate non-zero coefficinets
                    do ic=iX-1,iX+1
                        do jc=jY-1,jY+1
                            do kc=kZ-1,kZ+1
 
                                !Coefficients column possition
                                jcnt=(kc+nExternal+1)+(jc-1+nExternal+1)*nzTOT+(ic-1+nExternal+1)*nzTOT*nyTOT
                                
                                !Value of bfun in center of CV - for one point gauss integration - write this in array for computational speedup
                                CenterCVbfunValue=BFUN3D(nOrder,xVertex(ic),yVertex(jc),zVertex(kc),xCenter(ii),yCenter(jj),zCenter(kk),dx,dy,dz,0,0,0)
                                    
                                !Mass-matrix contribution
                                if(MassLump3D) then
                                    if(jcnt.eq.icnt) then
                                        Sij=SpecificStorage(IT)*Saturation(IT,Se_C)*CVvolume*PartOfConstantValue**3
                                        Cij=SpecMoistCapacity(IT,Se_C,SeModes_C)*CVvolume*PartOfConstantValue**3
                                    else
                                        Sij=0.d0
                                        Cij=0.d0
                                    endif
                                else
                                    Sij=SpecificStorage(IT)*Saturation(IT,Se_C)*CenterCVbfunValue*CVvolume
                                    Cij=SpecMoistCapacity(IT,Se_C,SeModes_C)*CenterCVbfunValue*CVvolume
                                endif
                                
                                !Conductivity matrix contribution
                                BFUNe=BFUN3D(nOrder,xVertex(ic),yVertex(jc),zVertex(kc),xcp2(ii),yCenter(jj),zCenter(kk),dx,dy,dz,1,0,0)
                                BFUNw=BFUN3D(nOrder,xVertex(ic),yVertex(jc),zVertex(kc),xcp1(ii),yCenter(jj),zCenter(kk),dx,dy,dz,1,0,0)
                                BFUNn=BFUN3D(nOrder,xVertex(ic),yVertex(jc),zVertex(kc),xCenter(ii),ycp2(jj),zCenter(kk),dx,dy,dz,0,1,0)
                                BFUNs=BFUN3D(nOrder,xVertex(ic),yVertex(jc),zVertex(kc),xCenter(ii),ycp1(jj),zCenter(kk),dx,dy,dz,0,1,0)
                                BFUNt=BFUN3D(nOrder,xVertex(ic),yVertex(jc),zVertex(kc),xCenter(ii),yCenter(jj),zcp2(kk),dx,dy,dz,0,0,1)
                                BFUNb=BFUN3D(nOrder,xVertex(ic),yVertex(jc),zVertex(kc),xCenter(ii),yCenter(jj),zcp1(kk),dx,dy,dz,0,0,1)
                                Kij=Kije*BFUNe+Kijw*BFUNw+Kijn*BFUNn+Kijs*BFUNs+Kijt*BFUNt+Kijb*BFUNb
                                
                                !Contribution of Dirichlet b.c.
                                if(DirichletBC) then
                                    BFUNe=BFUN3D(nOrder,xVertex(ic),yVertex(jc),zVertex(kc),xcp2(ii),yCenter(jj),zCenter(kk),dx,dy,dz,0,0,0)
                                    BFUNw=BFUN3D(nOrder,xVertex(ic),yVertex(jc),zVertex(kc),xcp1(ii),yCenter(jj),zCenter(kk),dx,dy,dz,0,0,0)
                                    BFUNn=BFUN3D(nOrder,xVertex(ic),yVertex(jc),zVertex(kc),xCenter(ii),ycp2(jj),zCenter(kk),dx,dy,dz,0,0,0)
                                    BFUNs=BFUN3D(nOrder,xVertex(ic),yVertex(jc),zVertex(kc),xCenter(ii),ycp1(jj),zCenter(kk),dx,dy,dz,0,0,0)
                                    BFUNt=BFUN3D(nOrder,xVertex(ic),yVertex(jc),zVertex(kc),xCenter(ii),yCenter(jj),zcp2(kk),dx,dy,dz,0,0,0)
                                    BFUNb=BFUN3D(nOrder,xVertex(ic),yVertex(jc),zVertex(kc),xCenter(ii),yCenter(jj),zcp1(kk),dx,dy,dz,0,0,0)
                                    if(StronglyImposedDirichetBC3D) then
                                        Kij=0.d0; Sij=0.d0; Cij=0.d0
                                    endif
                                    Kij=Kij+KijDBCe*BFUNe+KijDBCw*BFUNw+KijDBCn*BFUNn+KijDBCs*BFUNs+KijDBCt*BFUNt+KijDBCb*BFUNb
                                endif

                                !Final aij coefficient value
                                coeff=cfSS3D*(Sij+Cij)+Kij
                                
                                !Part of RHS contribution from known jcnt coefficient
                                CC_t1(icnt)=CC_t1(icnt)+cfSS3D*(CC_LastIter(jcnt)*Cij+CC_t0(jcnt)*Sij)
                                                                  
                                !Put value into global matrix array
                                if(LinSol.eq.'simq') then
                                    ASIMQ(icnt,jcnt)=coeff
                                elseif(LinSol.eq.'paradiso') then
                                    JA(icntNNZ)=jcnt
                                    APARADISO(icntNNZ)=coeff
                                    if(icnt.eq.jcnt) IURjcnt=icntNNZ  !Remember for implicit under-relaxation
                                    icntNNZ=icntNNZ+sup1
                                endif
                          
                            enddo
                        enddo
                    enddo
                    
                    !RIGHT HAND SIDE VECTOR
                    OOWi=OOWijFUN(IT,Se_C,NUR,NUR,NUR,xCenter(ii),yCenter(jj),zCenter(kk))*CVvolume
                    CC_t1(icnt)=CC_t1(icnt)+OOWi-dt3D*ExchangeSourceTerm(xcp1(ii),xcp2(ii),ConduitEx)
                    !Overwrite RHS when strong imposition of Dirichlet b.c.
                    if(DirichletBC.AND.StronglyImposedDirichetBC3D) CC_t1(icnt)=SIDbcValue
                    
                    !Implicit Under-Relaxation
                    if(ImplicitUnderRelax.ne.1.d0) then
                        if(linsol.eq.'simq') then
                            CC_t1(icnt)=CC_t1(icnt)+(1.d0-ImplicitUnderRelax)/ImplicitUnderRelax*ASIMQ(icnt,icnt)*CC_LastIter(icnt)
                            ASIMQ(icnt,icnt)=ASIMQ(icnt,icnt)/ImplicitUnderRelax
                        elseif(LinSol.eq.'paradiso') then
                            CC_t1(icnt)=CC_t1(icnt)+(1.d0-ImplicitUnderRelax)/ImplicitUnderRelax*APARADISO(IURjcnt)*CC_LastIter(icnt)
                            APARADISO(IURjcnt)=APARADISO(IURjcnt)/ImplicitUnderRelax
                        endif
                    endif   

                ENDDO
            ENDDO
        ENDDO                    
        
        
    end subroutine

    subroutine SolveSystemOfEquations
    !Solve discretized system of equations

        integer (kind=4) ii,jj,SimqKS,icnt
        !real (kind=8) cpuTS,cpuTE
        
        !Write system of equations into data file - full direct solver
        if(linsol.eq.'simq'.AND.writeSoE.eq.'yes') then
            open(lunF1,file='Ax=b_RE3D.dat')
            do ii=1,NTOT3D
                do jj=1,NTOT3D    
                    write(lunF1,'(e30.20,$)') ASIMQ(ii,jj)
                enddo
                write(lunF1,'(e30.20)') CC_t1(ii)
            enddo
            close(lunF1)
        endif

        !Write system of equations into data files - sparse direct solver
        if(LinSol.eq.'paradiso'.AND.writeSoE.eq.'yes') then
            open(lunf1,file='Aparadiso.dat')
            open(lunf2,file='JAparadiso.dat')
            open(lunf3,file='NNZparadiso.dat')
            open(lunf4,file='IAparadiso.dat')
            write(lunf1,'(e30.20,$)') aparadiso(1)
            write(lunf2,'(i7,$)') JA(1)
            write(lunf3,'(i7,$)') 1
            ii=2
            do icntNNZ=2,NNZ
                !Go to next line
                if(IA(ii).eq.icntNNZ) then
                    ii=ii+1
                    write(lunf1,'(/,e30.20,$)') aparadiso(icntNNZ)
                    write(lunf2,'(/,i7,$)') JA(icntNNZ)
                    write(lunf3,'(/,i7,$)') icntNNZ
                else
                    write(lunf1,'(e30.20,$)') aparadiso(icntNNZ)
                    write(lunf2,'(i7,$)') JA(icntNNZ)
                    write(lunf3,'(i7,$)') icntNNZ
                endif    
            enddo
            do ii=1,NTOT3D
               write(lunf4,'(i7)') IA(ii)
            enddo
            IA(ii)=NNZ+1
            close(lunF1)
            close(lunF2)
            close(lunF3)
            close(lunF4)
        endif
!call CPU_TIME(cpuTS)
        !Solve system of equation - full direct solver
        if(LinSol.eq.'simq') call Simq(ASIMQ,CC_t1,NTOT3D,SimqKS)
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) write(*,*) 'SIMQ singularity.'
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) read (*,*) 
        
        !Solve system of equation - sparse direct solver
        if(LinSol.eq.'paradiso') IA(NTOT3D+1)=NNZ+1
        if(LinSol.eq.'paradiso') call pardiso_sym_f90(NTOT3D,NNZ,IA,JA,APARADISO,CC_t1)
!call CPU_TIME(cpuTE)
!write(*,*) 'CPU time (paradiso):',cpuTE-cpuTS
        !Explicit Under-Relaxation
        if(ExplicitUnderRelax.ne.1.d0) then
            do icnt=1,NTOT3D
                CC_t1(icnt)=CC_LastIter(icnt)+ExplicitUnderRelax*(CC_t1(icnt)-CC_LastIter(icnt))
            enddo
        endif
                     
    end subroutine
    
    subroutine CheckConvergence(eps,icntEps,coorEps)
    
        integer(kind=4) ii,jj,kk,icnt,icntEps,coorEps(3)
        real(kind=8) eps,epsTemp
        
        !Skip first iteration
        if(iterKFM.eq.1.AND.iter.eq.1) then
            eps=eps_large
            return
        endif
        
        !Set to zero
        eps=0.d0
        
        !Convergence criterium based on absolute difference of solution coefficients 
        !do ii=1,NTOT3D
        !    epsTemp=eps
        !    eps=max(eps,abs(CC_t1(ii)-CC_lastIter(ii)))
        !    if(epsTemp.ne.eps) icntEps=ii
        !enddo
        icnt=0
        do ii=-nExternal,nx-1+nExternal
            do jj=-nExternal,ny-1+nExternal
                do kk=-nExternal,nz-1+nExternal
                    icnt=icnt+1
                    epsTemp=eps
                    eps=max(eps,abs(CC_t1(icnt)-CC_lastIter(icnt)))
                    if(epsTemp.ne.eps) then
                        icntEps=icnt
                        coorEps(1:3)=(/ii,jj,kk/)
                        CoeffEps3D=CC_t1(icntEps)
                        ValueEps3D=CXYZ(xCenter(ii),yCenter(jj),zCenter(kk),0,0,0,CC_t1)                        
                    endif
                enddo            
            enddo    
        enddo        
        
        if(CheckResidual.eq.'no') return
        
        !CheckResidual - missing
        
        !!Additional convergence criterium based on absolute difference of solution values
        !if(eps.le.eps_nonlin) then
        !    do ii=2,nn-1
        !        xp=xv(ii)
        !        eps=max(eps,abs(cx(xp,0,cc_t1)-cx(xp,0,cc_lastiter)))
        !    enddo
        !endif

    
    end subroutine


subroutine CheckConvergence_Transport(eps,icntEps,coorEps)
    
        integer(kind=4) ii,jj,kk,icnt,icntEps,coorEps(3)
        real(kind=8) eps,epsTemp
        
!        !Skip first iteration
!        if(iterKFM.eq.1.AND.iter.eq.1) then
!            eps=eps_large
!            return
!        endif
        
        !Set to zero
        eps=0.d0
        
        !Convergence criterium based on absolute difference of solution coefficients 
        !do ii=1,NTOT3D
        !    epsTemp=eps
        !    eps=max(eps,abs(CC_t1(ii)-CC_lastIter(ii)))
        !    if(epsTemp.ne.eps) icntEps=ii
        !enddo
        icnt=0
        do ii=-nExternal,nx-1+nExternal
            do jj=-nExternal,ny-1+nExternal
                do kk=-nExternal,nz-1+nExternal
                    icnt=icnt+1
                    epsTemp=eps
                    eps=max(eps,abs(CCC_t1(icnt)-CCC_LastIter(icnt)))
                    if(epsTemp.ne.eps) then
                        icntEps=icnt
                        coorEps(1:3)=(/ii,jj,kk/)
 !                       CoeffEps3D=CC_t1(icntEps)
 !                       ValueEps3D=CXYZ(xCenter(ii),yCenter(jj),zCenter(kk),0,0,0,CC_t1)                        
                    endif
                enddo            
            enddo    
        enddo        
        
 !       if(CheckResidual.eq.'no') return
        
        !CheckResidual - missing
        
        !!Additional convergence criterium based on absolute difference of solution values
        !if(eps.le.eps_nonlin) then
        !    do ii=2,nn-1
        !        xp=xv(ii)
        !        eps=max(eps,abs(cx(xp,0,cc_t1)-cx(xp,0,cc_lastiter)))
        !    enddo
        !endif

    
    end subroutine

    
    subroutine TestSoilProperties
    
        integer (kind=4) np,iSoilType
        real(kind=8) pp,pmin,pmax,dp
        real(kind=8) Se,O,kr,C,S(2),K
        
        !Test soil
        iSoilType=1
        
        pmin=0.d0
        pmax=10.d0
        np=10000

        open(lunF1,file='TestSoilProperties_p.dat')
        write(lunF1,'(a)') 'TITLE = TestSoilProperties_P"'
        write(lunF1,'(a)')  'VARIABLES = "p", "K", "O", "C"'
        
        open(lunF2,file='TestSoilProperties_O.dat')
        write(lunF2,'(a)') 'TITLE = TestSoilProperties_O"'
        write(lunF2,'(a)')  'VARIABLES = "O", "K"'
    
        dp=(pmax-pmin)/float(np)
        do pp=pmin,pmax,dp
            Se=EffectiveSaturation(iSoilType,-pp,S)
            kr=RelativePermeability(iSoilType,Se,S)
            O=WaterContent(iSoilType,Se)
            C=SpecMoistCapacity(iSoilType,Se,S)
            K=kr*SatCond(iSoilType)
            write(lunF1,'(4e20.10)') pp,K,O,C
            write(lunF2,'(2e20.10)') O,K
        enddo
        close(lunF1)
        close(lunF2)

                
    end subroutine    
    
    subroutine WriteConductivityField
    
        integer (kind=4) ii,jj,kk,it
        real (kind=8) xp,yp,zp
        real (kind=8) Ks
        
        
            !Ks field
            open(lunF1,file='Ks_Field.dat')
            write(lunF1,'(a)') 'TITLE = FFVT_IC_Matrix "'
            write(lunF1,'(a)')  'VARIABLES = "X", "Y", "Z", "K [m/s]"'
            write(lunF1,'(4(2xa,i4))') 'ZONE I=',nz,', J=',ny,', K=',nx,'F=POINT'
               
            do ii=0,nx-1
                do jj=0,ny-1
                    do kk=0,nz-1
                        xp=xVertex(ii)
                        yp=yVertex(jj)
                        zp=zVertex(kk)
                        it=SoilType(min(ii,nx),min(jj,ny),min(kk,nz))
                        Ks=SatCond(it)
                        write(lunF1,'(4e20.10)') xp,yp,zp,Ks
                    enddo
                enddo
            enddo 
            close(lunF1)        

    end subroutine    
      
!======================================================!
        
    subroutine FVGeometry(ll,nl,dll1,dll2,dl,lVertex,lCenter,lcp1,lcp2,indl)
    !Define finite volume geometry for specific "l" direction. Fup1 or B2.
    !Input: ll,nl,dll1,dll2,dl
    !Output: lVertex,lCenter,lcp1,lcp2,indl
    implicit none

        integer (kind=4) ll,nl,indl
        real (kind=8) dll1,dll2,dl,lVertex,lCenter,lcp1,lcp2
        
        !Basis function vertex
        lVertex=dll1+dfloat(ll)*dl+dl/2.d0
        !Finite volume centers
        indl=0        
        lCenter=lVertex
        if(ll.eq.-nExternal) then
            indl=-2
            lCenter=dll1+0.25d0*dl
        endif
        if(ll.eq.-nExternal+1) then
            indl=-1
            lCenter=dll1+0.75d0*dl
        endif
        if(ll.eq.nl-1) then
            indl=+1
            lCenter=dll2-0.75d0*dl
        endif
        if(ll.eq.nl-1+nExternal) then
            indl=+2
            lCenter=dll2-0.25d0*dl
        endif
        !Finite volume boundaries
        lcp1=lCenter-0.5d0*dl
        lcp2=lCenter+0.5d0*dl
        !Modify first/last two CVs
        if(ll.le.-nExternal+1.OR.ll.ge.nl-1) then
            lcp1=lCenter-0.25d0*dl
            lcp2=lCenter+0.25d0*dl
        endif
        !il=idnint((lCenter-0.5d0*dl-dll1)/(dl))
        
        
        
    
    end subroutine        
             
    subroutine ConductivityMatrixContribution(ii,jj,kk,CVdx,CVdy,CVdz,Kije,Kijw,Kijn,Kijs,Kijt,Kijb)
    !Uses last Picard iteration values to calculate conductivity term conribution.
    !Kij=-dt*int(K*div(BFUN_j)*n)dGama_i
    
        integer (kind=4) ii,jj,kk
        integer (kind=4) itC,itE,itW,itN,itS,itT,itB,itUSW
        real (kind=8) xc,yc,zc,CVdx,CVdy,CVdz
        real (kind=8) xE,xW,yN,yS,zT,zB
        real (kind=8) hC,hE,hW,hN,hS,hT,hB
        real (kind=8) Kije,Kijw,Kijn,Kijs,Kijt,Kijb
        real (kind=8) aKt,aKb
        real (kind=8) KsC,KsE,KsW,KsN,KsS,KsT,KsB
        real (kind=8) kr_C,kr_e,kr_w,kr_n,kr_s,kr_t,kr_b
        real (kind=8) Se_C,Se_E,Se_W,Se_N,Se_S,Se_T,Se_B
        real (kind=8) SeModes_C(2),SeModes_E(2),SeModes_W(2),SeModes_N(2),SeModes_S(2),SeModes_T(2),SeModes_B(2)
        
        !CV center position
        xc=xCenter(ii)
        yc=yCenter(jj)
        zc=zCenter(kk)

        !Neigbor CV values
        !X
        if(ii.ne.nx) then
            xE=xCenter(ii+1)
        else
            xE=dlx2
        endif
                if(ii.ne.-nExternal) then
                    xW=xCenter(ii-1)
                else
                    xW=dlx1
                endif
        !Y
        if(jj.ne.ny) then
            yN=yCenter(jj+1)
        else
            yN=dly2
        endif
                if(jj.ne.-nExternal) then
                    yS=yCenter(jj-1)
                else
                    yS=dly1
                endif
        !Z
        if(kk.ne.nz) then
            zT=zCenter(kk+1)
        else
            zT=dlz2
        endif
                if(kk.ne.-nExternal) then
                    zB=zCenter(kk-1)
                else
                    zB=dlz1
                endif
                
        !Hydraulic head values in center of each neigbor CV
        hC=CXYZ(xc,yc,zc,0,0,0,CC_LastIter)
        hE=CXYZ(xE,yc,zc,0,0,0,CC_LastIter)
        hW=CXYZ(xW,yc,zc,0,0,0,CC_LastIter)
        hN=CXYZ(xc,yN,zc,0,0,0,CC_LastIter)
        hS=CXYZ(xc,yS,zc,0,0,0,CC_LastIter)
        hT=CXYZ(xc,yc,zT,0,0,0,CC_LastIter)
        hB=CXYZ(xc,yc,zB,0,0,0,CC_LastIter)        
        
        !Soil material type
        itC=SoilType(ii,jj,kk)
        itE=SoilType(min(ii+1,nx),jj,kk)  
        itW=SoilType(max(ii-1,-nExternal),jj,kk)                   
        itN=SoilType(ii,min(jj+1,ny),kk)                   
        itS=SoilType(ii,max(jj-1,-nExternal),kk)                   
        itT=SoilType(ii,jj,min(kk+1,nz))                   
        itB=SoilType(ii,jj,max(kk-1,-nExternal))         
        
        !Saturated hydraulic conductivity for each neigbor CV
        KsC=SatCond(itC)
        KsE=SatCond(itE)  
        KsW=SatCond(itW)                   
        KsN=SatCond(itN)            
        KsS=SatCond(itS)                   
        KsT=SatCond(itT)            
        KsB=SatCond(itB)
        
        !Saturated interface conductivities - upstream weighting or harmonic mean
        if(UpStreamW_Ksat3D) then
            if(hC.ge.hE) Kije=KsC
            if(hC.lt.hE) Kije=KsE
            if(hC.ge.hW) Kijw=KsC
            if(hC.lt.hW) Kijw=KsW
            if(hC.ge.hN) Kijn=KsC
            if(hC.lt.hN) Kijn=KsN
            if(hC.ge.hS) Kijs=KsC
            if(hC.lt.hS) Kijs=KsS
            if(hC.ge.hT) Kijt=KsC/AnisotropZ(itC)
            if(hC.lt.hT) Kijt=KsT/AnisotropZ(itT)
            if(hC.ge.hB) Kijb=KsC/AnisotropZ(itC)
            if(hC.lt.hB) Kijb=KsB/AnisotropZ(itB)
        else
            Kije=2.d0*(KsC*KsE)/(KsC+KsE)
            Kijw=2.d0*(KsC*KsW)/(KsC+KsW)
            Kijn=2.d0*(KsC*KsN)/(KsC+KsN)
            Kijs=2.d0*(KsC*KsS)/(KsC+KsS)
            Kijt=2.d0*(KsC*KsT)/(KsC+KsT)
            Kijb=2.d0*(KsC*KsB)/(KsC+KsB) 
            !Z direction anisotropy
            aKt=2.d0*(AnisotropZ(itC)*AnisotropZ(itT))/(AnisotropZ(itC)+AnisotropZ(itT))
            aKb=2.d0*(AnisotropZ(itC)*AnisotropZ(itB))/(AnisotropZ(itC)+AnisotropZ(itB))
            Kijt=Kijt/aKt
            Kijb=Kijb/aKb
        endif
        
        !Relative interface conductivities - upstream weighting or edge values
        if(UpStreamW_Krel3D) then
            Se_C=EffectiveSaturation(itC,hC-zc,SeModes_C)
            Se_E=EffectiveSaturation(itE,hE-zc,SeModes_E)
            Se_W=EffectiveSaturation(itW,hW-zc,SeModes_W)
            Se_N=EffectiveSaturation(itN,hN-zc,SeModes_N)
            Se_S=EffectiveSaturation(itS,hS-zc,SeModes_S)
            Se_T=EffectiveSaturation(itT,hT-zT,SeModes_T)
            Se_B=EffectiveSaturation(itB,hB-zB,SeModes_B)

            kr_C=RelativePermeability(itC,Se_C,SeModes_C)
            if(hC.ge.hE) kr_e=kr_C
            if(hC.lt.hE) kr_e=RelativePermeability(itE,Se_E,SeModes_E)
            if(hC.ge.hW) kr_w=kr_C
            if(hC.lt.hW) kr_w=RelativePermeability(itW,Se_W,SeModes_W)
            if(hC.ge.hN) kr_n=kr_C
            if(hC.lt.hN) kr_n=RelativePermeability(itN,Se_N,SeModes_N)
            if(hC.ge.hS) kr_s=kr_C
            if(hC.lt.hS) kr_s=RelativePermeability(itS,Se_S,SeModes_S)
            if(hC.ge.hT) kr_t=kr_C
            if(hC.lt.hT) kr_t=RelativePermeability(itT,Se_T,SeModes_T)
            if(hC.ge.hB) kr_b=kr_C
            if(hC.lt.hB) kr_b=RelativePermeability(itB,Se_B,SeModes_B)
        else
            !Use material with higher hydraulic head (upstream weighting for deteremine which material use for edge values)
            itUSW=itC
            if(hC.lt.hE) itUSW=itE
            Se_E=EffectiveSaturation(itUSW,CXYZ(xcp2(ii),yc,zc,0,0,0,CC_LastIter)-zc,SeModes_E)
            kr_e=RelativePermeability(itUSW,Se_E,SeModes_E)
            itUSW=itC
            if(hC.lt.hW) itUSW=itW
            Se_W=EffectiveSaturation(itUSW,CXYZ(xcp1(ii),yc,zc,0,0,0,CC_LastIter)-zc,SeModes_W)
            kr_w=RelativePermeability(itUSW,Se_W,SeModes_W)
            itUSW=itC
            if(hC.lt.hN) itUSW=itN
            Se_N=EffectiveSaturation(itUSW,CXYZ(xc,ycp2(jj),zc,0,0,0,CC_LastIter)-zc,SeModes_N)
            kr_n=RelativePermeability(itUSW,Se_N,SeModes_N)
            itUSW=itC
            if(hC.lt.hS) itUSW=itS
            Se_S=EffectiveSaturation(itUSW,CXYZ(xc,ycp1(jj),zc,0,0,0,CC_LastIter)-zc,SeModes_S)
            kr_s=RelativePermeability(itUSW,Se_S,SeModes_S)
            itUSW=itC
            if(hC.lt.hT) itUSW=itT
            Se_T=EffectiveSaturation(itUSW,CXYZ(xc,yc,zcp2(kk),0,0,0,CC_LastIter)-zcp2(kk),SeModes_T)
            kr_t=RelativePermeability(itUSW,Se_T,SeModes_T)
            itUSW=itC
            if(hC.lt.hB) itUSW=itB
            Se_B=EffectiveSaturation(itUSW,CXYZ(xc,yc,zcp1(kk),0,0,0,CC_LastIter)-zcp1(kk),SeModes_B)
            kr_b=RelativePermeability(itUSW,Se_B,SeModes_B)       
        endif
        
        !Contribution from each edge of current CV - multiply with BFUN contribution and summ for final conductivity coefficient.
        Kije=-dt3D*kr_e*Kije*CVdy*CVdz
        Kijw=dt3D*kr_w*Kijw*CVdy*CVdz
        Kijn=-dt3D*kr_n*Kijn*CVdx*CVdz
        Kijs=dt3D*kr_s*Kijs*CVdx*CVdz
        Kijt=-dt3D*kr_t*Kijt*CVdx*CVdy
        Kijb=dt3D*kr_b*Kijb*CVdx*CVdy
                    
    
    end subroutine        
                                            
    subroutine BoundaryConditionsModifications(side,ii,jj,kk,CVArea,Kij,KijDBC,RHS,DBC,RHS_StrongImpDirichlet)
    !Modifies discretized equation to incorporate boundary conditions. 
    !Weak imposition of Neumann b.c.
    !Both weak and strong imposition of Dirichlet b.c.

        integer (kind=4) side,ii,jj,kk
        real (kind=8) xc,yc,zc,CVArea
        real (kind=8) Kij,KijDBC,RHS,RHS_StrongImpDirichlet,hOut
        character(len=20) chBC
        logical DBC
        
        xc=xCenter(ii)
        yc=yCenter(jj)
        zc=zCenter(kk)
        
        !Modify values
        chBC=BCType(side)
        if(chBC.eq.'Reservoir') then
            if(zcp1(kk).lt.ReservoirLevels(side)) then
                chBC='Dirichlet'
            else
                chBC='Neumann'
            endif
        endif
        
        !Neumann boundary condition
        if(chBC.eq.'Neumann') then
            Kij=0.d0
            KijDBC=0.d0
            RHS=RHS+dt3D*FluxBC(side,xc,yc,zc)*CVArea
            if(side.eq.5) then
                Qrain=Qrain+FluxBC(side,xc,yc,zc)*CVArea
                !write(*,*) +FluxBC(side,xc,yc,zc),CVArea,Qrain
            endif
        endif
        !Dirichet boundary condition    - vrijedi u slucaju da 1 CV ima samo jednu stranicu s Dirichlet b.c.
        if(chBC.eq.'Dirichlet') then
            KijDBC=dt3D*CVArea*BCPenalty3D
            RHS=RHS+dt3D*ReservoirLevels(side)*CVArea*BCPenalty3D
            DBC=.true.
            !Remember RHS value because it will be final (total) RHS value for strong imposition of Dirichlet b.c.
            if(StronglyImposedDirichetBC3D) RHS_StrongImpDirichlet=dt3D*ReservoirLevels(side)*CVArea*BCPenalty3D
        endif
        
        !Outflow boundary condition - set as Dirichlet b.c.
        if(chBC.eq.'Outflow') then
            KijDBC=dt3D*CVArea*BCPenalty3D
            call OutflowBC(side,ii,jj,kk,hOut)
            RHS=RHS+dt3D*hOut*CVArea*BCPenalty3D
            DBC=.true.
            !Remember RHS value because it will be final (total) RHS value for strong imposition of Dirichlet b.c.
            if(StronglyImposedDirichetBC3D) RHS_StrongImpDirichlet=dt3D*hOut*CVArea*BCPenalty3D
        endif  
    
    end subroutine        
                                            
    subroutine OutflowBC(side,ii,jj,kk,hOut)
    !Calculates outflow b.c.

        integer (kind=4) side,ii,jj,kk
        real (kind=8) xc,yc,zc,zOut,pOut,hOut
        
        !Outflow boundary condition - set as Dirichlet b.c. by using previous CVs center node hydraulic head
        if(side.eq.1) then
            xc=xCenter(nx-1)
            yc=yCenter(jj)
            zc=zCenter(kk)
            zOut=zc
        elseif(side.eq.2) then
            xc=xCenter(0)
            yc=yCenter(jj)
            zc=zCenter(kk)
            zOut=zc            
        elseif(side.eq.3) then
            xc=xCenter(ii)
            yc=yCenter(ny-1)
            zc=zCenter(kk)
            zOut=zc            
        elseif(side.eq.4) then
            xc=xCenter(ii)
            yc=yCenter(0)
            zc=zCenter(kk)
            zOut=zc            
        elseif(side.eq.5) then
            xc=xCenter(ii)
            yc=yCenter(jj)
            zc=zCenter(nz-1)
            zOut=dlz2            
        elseif(side.eq.6) then
            xc=xCenter(ii)
            yc=yCenter(jj)
            zc=zCenter(0)
            zOut=dlz1             
        endif
        
        hOut=CXYZ(xc,yc,zc,0,0,0,CC_LastIter)
        
        !pOut=CXYZ(xc,yc,zc,0,0,0,CC_LastIter)-zc
        !hOut=zOut+pOut

    end subroutine        
 
    subroutine trap_int_1D(FF,nord,xv,dx,mdx,X1,X2,a,b,kmax,Integ)
    !1D trapezoidal rule integration
    implicit none
        real (kind=8), external:: FF
        real (kind=8) xv,yv,dx,dy,X1,X2
        real (kind=8) a,b,Integ
        integer (kind=4) nord,mdx,kmax
        !
        real (kind=8) hx,xx,sx,f
        integer (kind=8) i,j,icnt
        integer (kind=8) k,nk
        logical (kind=4) ipar

        !Basic step
        hx=(b-a) 
        !Sum of boundary points
        sx=FF(nord,xv,a,dx,mdx,X1,X2)+FF(nord,xv,b,dx,mdx,X1,X2)
    
        icnt=2
        do k=1,kmax
            nk=2**k
            hx=(b-a)/real(nk)
            do i=0,nk
                if(mod(i,2).eq.0) then
                    ipar=.true.
                else
                    ipar=.false.
                endif

                xx=a+i*hx
                if(ipar) cycle
                f=FF(nord,xv,xx,dx,mdx,X1,X2)
                icnt=icnt+1
  
                if(i.eq.0.or.i.eq.nk) then
                    sx=sx+f
                else
                    sx=sx+2*f
                endif
            enddo
        enddo
        
        Integ=sx*hx/2.d0
        
        !write(*,'(a,i12,f20.10)') 'Integral 1D',icnt,Integ
    
    end subroutine trap_int_1D       
             
    subroutine trap_int_line_integ_3Dfun(FF,x1,x2,y1,y2,z1,z2,xv,yv,zv,mdx,mdy,mdz,kmax,Integ)
    !1D line trapezoidal rule integration for 3D function
    !Second and third coordinate ("yy" and "z") are assumed constant

        real (kind=8), external:: FF
        real (kind=8) xv,yv,zv
        real (kind=8) x1,x2,y1,y2,z1,z2,Integ
        integer (kind=4) mdx,mdy,mdz,kmax
        !
        real (kind=8) hx,hy,hz,xx,yy,zz
        real (kind=8) a,b
        real (kind=8) sx,sy,sz,f
        real (kind=8), parameter:: eps_small=1d-9
        integer (kind=4) i,j,icnt
        integer (kind=4) k,nk
        integer (kind=4) intx,inty,intz
        logical (kind=4) ipar,jpar
        
        write(*,*) 'Check "trap_int_line_integ_3Dfun"'
        read(*,*)
        stop
        
        !!Basic step
        !hx=(x2-x1)      ;   intx=0  
        !hy=(y2-y1)      ;   inty=0 
        !hz=(z2-z1)      ;   intz=0 
        !
        !!Find which coordinate is constatnt
        !if(abs(hx).lt.eps_small) intx=1
        !if(abs(hy).lt.eps_small) inty=1
        !if(abs(hz).lt.eps_small) intz=1
        !
        !!Set appropriate boundaries
        !if(intx.eq.1.AND.inty.eq.1) then
        !    a=z1   
        !    b=z2 
        !    yy=x1
        !    zz=y1
        !elseif(intx.eq.1.AND.intz.eq.1) then
        !    a=y1  
        !    b=y2 
        !    yy=x1
        !    zz=z1
        !elseif(inty.eq.1.AND.intz.eq.1) then
        !    a=x1   
        !    b=x2 
        !    yy=y1
        !    zz=z1
        !endif
        !
        !!
        !hx=(b-a)
        !
        !!Sum of boundary points 
        !if(intx.eq.1.AND.inty.eq.1) sx=FF(EffectiveSaturation(CXYZ(yy,zz,a,0,0,0,CC_LastIter)-a),xv,yv,zv,yy,zz,a,mdx,mdy,mdz)+&
        !                               FF(EffectiveSaturation(CXYZ(yy,zz,b,0,0,0,CC_LastIter)-b),xv,yv,zv,yy,zz,b,mdx,mdy,mdz)
        !
        !if(intx.eq.1.AND.intz.eq.1) sx=FF(EffectiveSaturation(CXYZ(yy,a,zz,0,0,0,CC_LastIter)-zz),xv,yv,zv,yy,a,zz,mdx,mdy,mdz)+&
        !                               FF(EffectiveSaturation(CXYZ(yy,b,zz,0,0,0,CC_LastIter)-zz),xv,yv,zv,yy,b,zz,mdx,mdy,mdz)
        !
        !if(inty.eq.1.AND.intz.eq.1) sx=FF(EffectiveSaturation(CXYZ(a,yy,zz,0,0,0,CC_LastIter)-zz),xv,yv,zv,a,yy,zz,mdx,mdy,mdz)+&
        !                               FF(EffectiveSaturation(CXYZ(b,yy,zz,0,0,0,CC_LastIter)-zz),xv,yv,zv,b,yy,zz,mdx,mdy,mdz)
        !
        !
        !icnt=2
        !do k=1,kmax
        !    nk=2**k
        !    hx=(b-a)/real(nk)
        !    do i=0,nk
        !        if(mod(i,2).eq.0) then
        !            ipar=.true.
        !        else
        !            ipar=.false.
        !        endif
        !
        !        xx=a+i*hx
        !        if(ipar) cycle
        !        if(intx.eq.1.AND.inty.eq.1) f= FF(EffectiveSaturation(CXYZ(yy,zz,xx,0,0,0,CC_LastIter)-xx),xv,yv,zv,yy,zz,xx,mdx,mdy,mdz)
        !        if(intx.eq.1.AND.intz.eq.1) f= FF(EffectiveSaturation(CXYZ(yy,xx,zz,0,0,0,CC_LastIter)-zz),xv,yv,zv,yy,xx,zz,mdx,mdy,mdz)
        !        if(inty.eq.1.AND.intz.eq.1) f= FF(EffectiveSaturation(CXYZ(xx,yy,zz,0,0,0,CC_LastIter)-zz),xv,yv,zv,xx,yy,zz,mdx,mdy,mdz)
        !        icnt=icnt+1
        !
        !        if(i.eq.0.or.i.eq.nk) then
        !            sx=sx+f
        !        else
        !            sx=sx+2*f
        !        endif
        !    enddo
        !enddo
        !
        !Integ=sx*hx/2.d0
        !
        !!write(*,'(a,i12,f20.10)') 'Integral 1D',icnt,Integ
    
    end subroutine trap_int_line_integ_3Dfun    
     
    subroutine trap_int_surface_integ_3Dfun(FF,x1,x2,y1,y2,z1,z2,xv,yv,zv,mdx,mdy,mdz,kmax,Integ)
    !2D surface trapezoidal rule integration of 3D function. Used for integration of conductivty matrix.
    !Third coordinate ("z") is assumed constant
    !FF-function to integrate - external function
    !CC-solution coefficients for evaluations of velocities
    !(mdx,mdy,mdz)-partial derivative orders of basis function
    !x1,x2,...-integration boundaries
    !kmax-parameter for numerical integration precision
    !Integ-returns calculated value of integral
    implicit none

        real (kind=8), external:: FF
        real (kind=8) x1,x2,y1,y2,z1,z2,Integ
        integer (kind=4) norder,mdx,mdy,mdz,kmax
        !
        real (kind=8) hx,hy,hz,xx,yy,zz
        real (kind=8) a,b,c,d
        real (kind=8) xv,yv,zv
        real (kind=8) sx,sy,sz,f,pp,Se
        real (kind=8), parameter:: eps_small=1d-9
        integer (kind=4) i,j,icnt
        integer (kind=4) k,nk
        integer (kind=4) intx,inty,intz
        logical (kind=4) ipar,jpar
        
        write(*,*) 'Check "trap_int_surface_integ_3Dfun"'
        read(*,*)
        stop
        
        !!Basic step
        !hx=(x2-x1)      ;   intx=0  
        !hy=(y2-y1)      ;   inty=0 
        !hz=(z2-z1)      ;   intz=0 
        !
        !!Find which coordinate is constatnt
        !if(abs(hx).lt.eps_small) intx=1
        !if(abs(hy).lt.eps_small) inty=1
        !if(abs(hz).lt.eps_small) intz=1
        !
        !!Set appropriate boundaries
        !if(intx.eq.1) then
        !    a=y1    ;   c=z1
        !    b=y2    ;   d=z2
        !    zz=x1
        !elseif(inty.eq.1) then
        !    a=x1    ;   c=z1
        !    b=x2    ;   d=z2
        !    zz=y1
        !elseif(intz.eq.1) then
        !    a=x1    ;   c=y1
        !    b=x2    ;   d=y2
        !    zz=z1
        !endif
        !
        !!
        !hx=(b-a)
        !hy=(d-c)
        !
        !!Sum of boundary points
        !if(intx.eq.1) then
        !    sy=FF(EffectiveSaturation(CXYZ(zz,a,c,0,0,0,CC_LastIter)-c),xv,yv,zv,zz,a,c,mdx,mdy,mdz)+&
        !       FF(EffectiveSaturation(CXYZ(zz,b,c,0,0,0,CC_LastIter)-c),xv,yv,zv,zz,b,c,mdx,mdy,mdz)+&
        !       FF(EffectiveSaturation(CXYZ(zz,a,d,0,0,0,CC_LastIter)-d),xv,yv,zv,zz,a,d,mdx,mdy,mdz)+&
        !       FF(EffectiveSaturation(CXYZ(zz,b,d,0,0,0,CC_LastIter)-d),xv,yv,zv,zz,b,d,mdx,mdy,mdz)
        !elseif(inty.eq.1) then
        !    sy=FF(EffectiveSaturation(CXYZ(a,zz,c,0,0,0,CC_LastIter)-c),xv,yv,zv,a,zz,c,mdx,mdy,mdz)+&
        !       FF(EffectiveSaturation(CXYZ(b,zz,c,0,0,0,CC_LastIter)-c),xv,yv,zv,b,zz,c,mdx,mdy,mdz)+&
        !       FF(EffectiveSaturation(CXYZ(a,zz,d,0,0,0,CC_LastIter)-d),xv,yv,zv,a,zz,d,mdx,mdy,mdz)+&
        !       FF(EffectiveSaturation(CXYZ(b,zz,d,0,0,0,CC_LastIter)-d),xv,yv,zv,b,zz,d,mdx,mdy,mdz)
        !elseif(intz.eq.1) then
        !    sy=FF(EffectiveSaturation(CXYZ(a,c,zz,0,0,0,CC_LastIter)-zz),xv,yv,zv,a,c,zz,mdx,mdy,mdz)+&
        !       FF(EffectiveSaturation(CXYZ(b,c,zz,0,0,0,CC_LastIter)-zz),xv,yv,zv,b,c,zz,mdx,mdy,mdz)+&
        !       FF(EffectiveSaturation(CXYZ(a,d,zz,0,0,0,CC_LastIter)-zz),xv,yv,zv,a,d,zz,mdx,mdy,mdz)+&
        !       FF(EffectiveSaturation(CXYZ(b,d,zz,0,0,0,CC_LastIter)-zz),xv,yv,zv,b,d,zz,mdx,mdy,mdz)
        !endif
        !
        !                                            
        !icnt=4
        !do k=1,kmax
        !    nk=2**k
        !    hx=(b-a)/real(nk)
        !    hy=(d-c)/real(nk)
        !    do i=0,nk
        !        if(mod(i,2).eq.0) then
        !            ipar=.true.
        !        else
        !            ipar=.false.
        !        endif
        !        yy=c+i*hy
        !        sx=0
        !        do j=0,nk
        !            if(mod(j,2).eq.0) then
        !                jpar=.true.
        !            else
        !                jpar=.false.
        !            endif
        !            xx=a+j*hx
        !            if(ipar.and.jpar) cycle
        !            !Function value
        !            if(intx.eq.1) f=FF(EffectiveSaturation(CXYZ(zz,xx,yy,0,0,0,CC_LastIter)-yy),xv,yv,zv,zz,xx,yy,mdx,mdy,mdz)
        !            if(inty.eq.1) f=FF(EffectiveSaturation(CXYZ(xx,zz,yy,0,0,0,CC_LastIter)-yy),xv,yv,zv,xx,zz,yy,mdx,mdy,mdz)
        !            if(intz.eq.1) f=FF(EffectiveSaturation(CXYZ(xx,yy,zz,0,0,0,CC_LastIter)-zz),xv,yv,zv,xx,yy,zz,mdx,mdy,mdz)
        !            icnt=icnt+1
        !            if(j.eq.0.or.j.eq.nk) then
        !                sx=sx+f
        !            else
        !                sx=sx+2*f
        !            endif
        !        enddo
        !        if(i.eq.0.or.i.eq.nk) then
        !            sy=sy+sx
        !        else
        !            sy=sy+2*sx
        !        endif
        !    enddo
        !enddo
        !
        !Integ=sy*hx*hy/4.d0
        !
        !!write(*,'(a,i12,f20.10)') 'Integral 2D',icnt,sy*hx*hy/4.
    
    end subroutine  
          
    subroutine trap_int_surface_integ_3Dfun_MatrixVelocities(FF,CC,mdx,mdy,mdz,x1,x2,y1,y2,z1,z2,kmax,Integ)
    !2D surface trapezoidal rule integration of 3D function. Used for integration of matrix velocites.
    !Third coordinate ("z") is assumed constant
    !FF-function to integrate - external function
    !CC-solution coefficients for evaluations of velocities
    !(mdx,mdy,mdz)-partial derivative orders of basis function
    !x1,x2,...-integration boundaries
    !kmax-parameter for numerical integration precision
    !Integ-returns calculated value of integral
    implicit none

        real (kind=8), external:: FF
        real (kind=8) x1,x2,y1,y2,z1,z2,Integ
        integer (kind=4) norder,mdx,mdy,mdz,kmax
        !
        real (kind=8) hx,hy,hz,xx,yy,zz,CC(*)
        real (kind=8) a,b,c,d
        real (kind=8) sx,sy,sz,f
        real (kind=8), parameter:: eps_small=1d-9
        integer (kind=4) i,j,icnt
        integer (kind=4) k,nk
        integer (kind=4) intx,inty,intz
        logical (kind=4) ipar,jpar

        write(*,*) 'Check "trap_int_surface_integ_3Dfun_MatrixVelocities"'
        read(*,*)
        stop        

        !!Basic step
        !hx=(x2-x1)      ;   intx=0  
        !hy=(y2-y1)      ;   inty=0 
        !hz=(z2-z1)      ;   intz=0 
        !
        !!Find which coordinate is constatnt
        !if(abs(hx).lt.eps_small) intx=1
        !if(abs(hy).lt.eps_small) inty=1
        !if(abs(hz).lt.eps_small) intz=1
        !
        !!Set appropriate boundaries
        !if(intx.eq.1) then
        !    a=y1    ;   c=z1
        !    b=y2    ;   d=z2
        !    zz=x1
        !elseif(inty.eq.1) then
        !    a=x1    ;   c=z1
        !    b=x2    ;   d=z2
        !    zz=y1
        !elseif(intz.eq.1) then
        !    a=x1    ;   c=y1
        !    b=x2    ;   d=y2
        !    zz=z1
        !endif
        !
        !!
        !hx=(b-a)
        !hy=(d-c)
        !                                !(xp,yp,zp,mdx,mdy,mdz,CC)
        !!Sum of boundary points 
        !if(intx.eq.1) sy=FF(zz,a,c,mdx,mdy,mdz,CC)+FF(zz,a,d,mdx,mdy,mdz,CC)+ &
        !                 FF(zz,b,c,mdx,mdy,mdz,CC)+FF(zz,b,d,mdx,mdy,mdz,CC)
        !if(inty.eq.1) sy=FF(a,zz,c,mdx,mdy,mdz,CC)+FF(a,zz,d,mdx,mdy,mdz,CC)+ &
        !                 FF(b,zz,c,mdx,mdy,mdz,CC)+FF(b,zz,d,mdx,mdy,mdz,CC)   
        !if(intz.eq.1) sy=FF(a,c,zz,mdx,mdy,mdz,CC)+FF(a,d,zz,mdx,mdy,mdz,CC)+ &
        !                 FF(b,c,zz,mdx,mdy,mdz,CC)+FF(b,d,zz,mdx,mdy,mdz,CC) 
        !
        !icnt=4
        !do k=1,kmax
        !    nk=2**k
        !    hx=(b-a)/real(nk)
        !    hy=(d-c)/real(nk)
        !    do i=0,nk
        !        if(mod(i,2).eq.0) then
        !            ipar=.true.
        !        else
        !            ipar=.false.
        !        endif
        !        yy=c+i*hy
        !        sx=0
        !        do j=0,nk
        !            if(mod(j,2).eq.0) then
        !                jpar=.true.
        !            else
        !                jpar=.false.
        !            endif
        !            xx=a+j*hx
        !            if(ipar.and.jpar) cycle
        !            !Function value
        !            if(intx.eq.1) f=FF(zz,xx,yy,mdx,mdy,mdz,CC)
        !            if(inty.eq.1) f=FF(xx,zz,yy,mdx,mdy,mdz,CC)
        !            if(intz.eq.1) f=FF(xx,yy,zz,mdx,mdy,mdz,CC)
        !            icnt=icnt+1
        !            if(j.eq.0.or.j.eq.nk) then
        !                sx=sx+f
        !            else
        !                sx=sx+2*f
        !            endif
        !        enddo
        !        if(i.eq.0.or.i.eq.nk) then
        !            sy=sy+sx
        !        else
        !            sy=sy+2*sx
        !        endif
        !    enddo
        !enddo
        !
        !Integ=sy*hx*hy/4.d0
        !
        !!write(*,'(a,i12,f20.10)') 'Integral 2D',icnt,sy*hx*hy/4.
    
    end subroutine  
             
    subroutine trap_int_3D(FF,a,b,c,d,e,f,xv,yv,zv,lmax,Integ)
    !3D trapezoidal rule integration on rectangular cuboid.
    !Used for 3D mass matrix integrals.
    implicit none
    real (kind=8), external:: FF
    real (kind=8) a,b,c,d,e,f
    real (kind=8) xv,yv,zv
    real (kind=8) pp,Se
    real (kind=8) hx,hy,hz,xx,yy,zz,sx,sy,sz,fv,Integ
    integer i,j,k,icnt
    integer l,lmax,nl
    logical ipar,jpar,kpar

        write(*,*) 'Check "trap_int_3D"'
        read(*,*)
        stop 
    
    !!Integration boundaries, x e [a,b], y e [c,d], z e [e,f]
    !
    !!Basic step
    !hx=(b-a) 
    !hy=(d-c)
    !hz=(f-e)
    !!Sum of boundary values
    !sz=0.d0
    !sz=FF(EffectiveSaturation(CXYZ(a,c,e,0,0,0,CC_LastIter)-e),xv,yv,zv,a,c,e)+ &
    !   FF(EffectiveSaturation(CXYZ(a,d,e,0,0,0,CC_LastIter)-e),xv,yv,zv,a,d,e)+ &
    !   FF(EffectiveSaturation(CXYZ(b,c,e,0,0,0,CC_LastIter)-e),xv,yv,zv,b,c,e)+ &
    !   FF(EffectiveSaturation(CXYZ(b,d,e,0,0,0,CC_LastIter)-e),xv,yv,zv,b,d,e)+ &
    !   FF(EffectiveSaturation(CXYZ(a,c,f,0,0,0,CC_LastIter)-f),xv,yv,zv,a,c,f)+ &
    !   FF(EffectiveSaturation(CXYZ(a,d,f,0,0,0,CC_LastIter)-f),xv,yv,zv,a,d,f)+ &
    !   FF(EffectiveSaturation(CXYZ(b,c,f,0,0,0,CC_LastIter)-f),xv,yv,zv,b,c,f)+ &
    !   FF(EffectiveSaturation(CXYZ(b,d,f,0,0,0,CC_LastIter)-f),xv,yv,zv,b,d,f)
    !
    !icnt=8
    !do l=1,lmax
    !    nl=2**l
    !    hx=(b-a)/real(nl)
    !    hy=(d-c)/real(nl)
    !    hz=(f-e)/real(nl)
    !
    !    !z-direction
    !    do k=0,nl
    !        if(mod(k,2).eq.0) then
    !            kpar=.true.
    !        else
    !            kpar=.false.
    !        endif        
    !        zz=e+k*hz
    !        sy=0.d0
    !        !y-direction
    !        do j=0,nl
    !            if(mod(j,2).eq.0) then
    !                jpar=.true.
    !            else
    !                jpar=.false.
    !            endif
    !            yy=c+j*hy
    !            sx=0.d0
    !            !x-direction
    !            do i=0,nl
    !                if(mod(i,2).eq.0) then
    !                    ipar=.true.
    !                else
    !                    ipar=.false.
    !                endif
    !                xx=a+i*hx
    !                !Calculate functions value in integration point (xx,yy,zz)
    !                if(ipar.and.jpar.and.kpar) cycle
    !                icnt=icnt+1
    !                pp=CXYZ(xx,yy,zz,0,0,0,CC_LastIter)-zz
    !                Se=EffectiveSaturation(pp)
    !                fv=FF(Se,xv,yv,zv,xx,yy,zz)
    !                if(i.eq.0.or.i.eq.nl) then
    !                    sx=sx+fv
    !                else
    !                    sx=sx+2*fv
    !                endif
    !            enddo
    !            if(j.eq.0.or.j.eq.nl) then
    !                sy=sy+sx
    !            else
    !                sy=sy+2*sx
    !            endif
    !        enddo
    !        if(k.eq.0.or.k.eq.nl) then
    !            sz=sz+sy
    !        else
    !            sz=sz+2*sy
    !        endif
    !    enddo
    !
    !enddo
    !
    !Integ=sz*hx*hy*hz/8.d0
    !
    !!write(*,'(a,2f20.10)') 'Integral 3D',Integ,(b-a)*(d-c)*(f-e)*PartOfConstantValue**3
    
    end subroutine trap_int_3D       
                  
    subroutine Simq(A,B,N,KS)
    !Solves full system of equations by Gauss elimination.

        IMPLICIT real(8) (A-H, O-Z)
        DIMENSION A(1),B(1)
        !
        integer (kind=4) jj,n,ks,j,jy,it,i,ij,imax,i1,k,i2,iqs,&
                         ix,ixj,jx,ixjx,jjx,ny,ia,ib,ic
        
        
    !C
    !C***  FORWARD RJESENJE
    !C
          TOL = 0.000000001q0
          KS = 0
          JJ = -N
          DO 65 J=1,N
          JY=J+1
          JJ=JJ+N+1
          BIGA = 0.0q0
          IT=JJ-J
          DO 30 I=J,N
    !C
    !C***  TRAZANJE MAKSIMALNOG KOEFICIJENTA U STUPCU
    !C
          IJ=IT+I
          IF(abs(BIGA)-abs(A(IJ)))  20,30,30
      20  BIGA = A(IJ)
          IMAX=I
      30  CONTINUE
    !C
    !C***  ISPITIVANJE DA LI JE PIVOT MANJI OD TOLERANCE (SINGULARNA MATRICA)
    !C
          IF(abs(BIGA)-TOL) 35,35,40
      35  KS=1
          RETURN
    !C
    !C***  MEDJUSOBNA ZAMJENA REDAKA AKO JE POTREBNO
    !C
      40  I1=J+N*(J-2)
          IT=IMAX-J
          DO 50 K=J,N
          I1=I1+N
          I2=I1+IT
          SAVE = A(I1)
          A(I1) = A(I2)
          A(I2) = SAVE
    !C
    !C***  DIJELJENJE JEDNADZBE S VODECIM KOEFICIJENTOM
    !C
      50  A(I1) = A(I1)/BIGA
          SAVE = B(IMAX)
          B(IMAX) = B(J)
          B(J) = SAVE/BIGA
    !C
    !C***  ELIMINIRANJE SLIJEDECE VARIJABLE
    !C
          IF(J-N) 55,70,55
      55  IQS=N*(J-1)
          DO 65 IX=JY,N
          IXJ=IQS+IX
          IT=J-IX
          DO 60 JX=JY,N
          IXJX = N*(JX-1)+IX
          JJX=IXJX+IT
      60  A(IXJX) = A(IXJX)-(A(IXJ)*A(JJX))
      65  B(IX) = B(IX)-(B(J)*A(IXJ))
    !C
    !C***  BACK SUPSTITUCIJA
    !C
      70  NY=N-1
          IT=N*N
          DO 80 J=1,NY
          IA=IT-J
          IB=N-J
          IC=N
          DO 80 K=1,J
          B(IB)=B(IB)-A(IA)*B(IC)
          IA=IA-N
      80  IC=IC-1
          RETURN

    end subroutine Simq

    subroutine pardiso_sym_f90(n,nnz,ia,ja,a,b)
    !Sparse matrix solver. MKL libraries.
        USE mkl_pardiso
        IMPLICIT NONE
        INTEGER (kind=4), PARAMETER :: dp = KIND(1.0D0)
        !.. Internal solver memory pointer 
        TYPE(MKL_PARDISO_HANDLE), ALLOCATABLE  :: pt(:)
        !.. All other variables
        INTEGER (kind=4) maxfct, mnum, mtype, phase, n, nrhs, error, msglvl, nnz
        INTEGER (kind=4) error1
        INTEGER (kind=4), ALLOCATABLE :: iparm( : )
        !INTEGER, ALLOCATABLE :: ia( : )
        !INTEGER, ALLOCATABLE :: ja( : )
        !REAL(KIND=DP), ALLOCATABLE :: a( : )
        !REAL(KIND=DP), ALLOCATABLE :: b( : )
        REAL(KIND=DP), ALLOCATABLE :: x( : )
        INTEGER (kind=4) i, idum(1)
        REAL(KIND=DP) ddum(1)
        !
        integer (kind=4) ia(n+1),ja(nnz)
        REAL(KIND=DP) a(nnz),b(n)!,x(n)
        !
        !.. Fill all arrays containing matrix data.
        !n = 8 
        !nnz = 18
        nrhs = 1 
        maxfct = 1 
        mnum = 1
        
!EXAMPLE IS FOR SYMMETRIC MATRIX

        !ALLOCATE( ia ( n + 1 ) )
        !ia = (/ 1, 5, 8, 10, 12, 15, 17, 18, 19 /)
        !ALLOCATE( ja ( nnz ) )
        !ja = (/ 1,    3,       6, 7,    &
        !           2, 3,    5,          &
        !              3,             8, &
        !                 4,       7,    &
        !                    5, 6, 7,    &
        !                       6,    8, &
        !                          7,    &
        !                             8 /)
        !ALLOCATE( a ( nnz ) )
        !a = (/ 7.d0,        1.d0,             2.d0, 7.d0,        &
        !             -4.d0, 8.d0,       2.d0,                    &
        !                    1.d0,                         5.d0,  &
        !                          7.d0,             9.d0,        &
        !                                5.d0, 1.d0, 5.d0,        &
        !                                     -1.d0,       5.d0,  &
        !                                           11.d0,        &
        !                                                  5.d0 /)
        !ALLOCATE( b ( n ) )
        ALLOCATE( x ( n ) )
        !..
        !.. Set up PARDISO control parameter
        !..
        ALLOCATE( iparm ( 64 ) )

        do i = 1, 64
           iparm(i) = 0
        end do 

        iparm(1) = 1 ! no solver default
        iparm(2) = 2 ! fill-in reordering from METIS
        iparm(4) = 0 ! no iterative-direct algorithm
        iparm(5) = 0 ! no user fill-in reducing permutation
        iparm(6) = 0 ! =0 solution on the first n compoments of x
        iparm(8) = 9 ! numbers of iterative refinement steps
        iparm(10) = 13 ! perturbe the pivot elements with 1E-13
        iparm(11) = 1 ! use nonsymmetric permutation and scaling MPS
        iparm(13) = 0 ! maximum weighted matching algorithm is switched-off (default for symmetric). Try iparm(13) = 1 in case of inappropriate accuracy
        iparm(14) = 0 ! Output: number of perturbed pivots
        iparm(18) = -1 ! Output: number of nonzeros in the factor LU
        iparm(19) = -1 ! Output: Mflops for LU factorization
        iparm(20) = 0 ! Output: Numbers of CG Iterations
    

        error  = 0 ! initialize error flag
        msglvl = 0 ! print statistical information
        mtype  = 11 ! real and nonsymmetric (mtype=11)
                    ! symmetric, indefinite (mtype=-2)
                    
        !.. Initiliaze the internal solver memory pointer. This is only
        ! necessary for the FIRST call of the PARDISO solver.

        ALLOCATE ( pt ( 64 ) )
        do i = 1, 64
           pt( i )%DUMMY =  0 
        end do

        !.. Reordering and Symbolic Factorization, This step also allocates
        ! all memory that is necessary for the factorization

        phase = 11 ! only reordering and symbolic factorization

        CALL pardiso (pt, maxfct, mnum, mtype, phase, n, a, ia, ja, &
        idum, nrhs, iparm, msglvl, ddum, ddum, error)
    
        !WRITE(*,*) 'Reordering completed ... '
        IF (error /= 0) THEN
           !WRITE(*,*) 'The following ERROR was detected: ', error
           GOTO 1000
        END IF
        !WRITE(*,*) 'Number of nonzeros in factors = ',iparm(18)
        !WRITE(*,*) 'Number of factorization MFLOPS = ',iparm(19)

        !.. Factorization.
        phase = 22 ! only factorization
        CALL pardiso (pt, maxfct, mnum, mtype, phase, n, a, ia, ja, &
        idum, nrhs, iparm, msglvl, ddum, ddum, error)
        !WRITE(*,*) 'Factorization completed ... '
        IF (error /= 0) THEN
           WRITE(*,*) 'The following ERROR was detected: ', error
           GOTO 1000
        ENDIF

        !.. Back substitution and iterative refinement
        iparm(8) = 2 ! max numbers of iterative refinement steps
        phase = 33 ! only factorization
        !do i = 1, n
        !   b(i) = 1.d0
        !end do
        CALL pardiso (pt, maxfct, mnum, mtype, phase, n, a, ia, ja, &
        idum, nrhs, iparm, msglvl, b, x, error)
        !WRITE(*,*) 'Solve completed ... '
        IF (error /= 0) THEN
           WRITE(*,*) 'The following ERROR was detected: ', error
           GOTO 1000
        ENDIF
        !WRITE(*,*) 'The solution of the system is '
        !DO i = 1, n
        !   WRITE(*,*) ' x(',i,') = ', x(i)
        !END DO
    
    
        !!!
        b=x
      
        1000 CONTINUE
        !.. Termination and release of memory
        phase = -1 ! release internal memory
        CALL pardiso (pt, maxfct, mnum, mtype, phase, n, ddum, idum, idum, &
        idum, nrhs, iparm, msglvl, ddum, ddum, error1)

        !IF ( ALLOCATED( ia ) )      DEALLOCATE( ia )
        !IF ( ALLOCATED( ja ) )      DEALLOCATE( ja )
        !IF ( ALLOCATED( a ) )       DEALLOCATE( a )
        !IF ( ALLOCATED( b ) )       DEALLOCATE( b )
        !IF ( ALLOCATED( x ) )       DEALLOCATE( x )
        IF ( ALLOCATED( iparm ) )   DEALLOCATE( iparm )

        IF (error1 /= 0) THEN
           WRITE(*,*) 'The following ERROR on release stage was detected: ', error1
           STOP 1
        ENDIF

        IF ( error /= 0 ) then
            write(*,*) 'error'  !***!
            read(*,*)
            STOP 1
        endif
    
        !STOP 0
    END SUBROUTINE
        
    subroutine ScreenRecord(sc)
    
        character(*) sc

        SELECT CASE(sc)
            
        CASE('CalculateParameters')
            
            write(*,*)
            write(*,*) '3D MATRIX PARAMETERS'
            write(*,*) 'NTOT3D',NTOT3D
            write(*,*) 'NNZ   ',NNZ
            write(*,'(2(1xa,f12.5,2x))') 'xM1 ',x1Matrix,'xM2 ',x2Matrix,'yM1 ',y1Matrix,'yM2 ',y2Matrix,'zM1 ',z1Matrix,'zM2 ',z2Matrix
            write(*,'(1xa,2xf12.5)') 'dx',dx
            write(*,'(1xa,2xf12.5)') 'dy',dy
            write(*,'(1xa,2xf12.5)') 'dz',dz
            write(*,'(1xa,3xe20.10)') 'eps3D',epsMatrix3D
            write(*,'(1xa,3xe20.10)') 'IUR3D',IUR3D
            write(*,'(1xa,3xe20.10)') 'EUR3D',EUR3D
            write(*,'(1xa,1xe20.10)') 'epsUR3D',epsUR3D
            write(*,'(1xa,i6)') 'maxiter3D',maxiter3D
            write(*,*) 
            
            write(lunFScreen,*)
            write(lunFScreen,*) '3D MATRIX PARAMETERS'
            write(lunFScreen,*) 'NTOT3D',NTOT3D
            write(lunFScreen,*) 'NNZ   ',NNZ
            write(lunFScreen,'(2(1xa,f12.5,2x))') 'xM1 ',x1Matrix,'xM2 ',x2Matrix,'yM1 ',y1Matrix,'yM2 ',y2Matrix,'zM1 ',z1Matrix,'zM2 ',z2Matrix
            write(lunFScreen,'(1xa,2xf12.5)') 'dx',dx
            write(lunFScreen,'(1xa,2xf12.5)') 'dy',dy
            write(lunFScreen,'(1xa,2xf12.5)') 'dz',dz
            write(lunFScreen,'(1xa,3xe20.10)') 'eps3D',epsMatrix3D
            write(lunFScreen,'(1xa,3xe20.10)') 'IUR3D',IUR3D
            write(lunFScreen,'(1xa,3xe20.10)') 'EUR3D',EUR3D
            write(lunFScreen,'(1xa,1xe20.10)') 'epsUR3D',epsUR3D
            write(lunFScreen,'(1xa,i6)') 'maxiter3D',maxiter3D
            write(lunFScreen,*) 

        CASE('Time')
            write(*,*) 'Current time:',TCurr,'Iterations:',iter
            
            write(lunFScreen,*) 'Current time:',TCurr,'Iterations:',iter
            
        CASE('Discharge')
            write(*,'(6(3xa,f12.8))') 'QMo:',QmatrixOutlet*DischargeUnitConvert,'QMi:',QmatrixInlet*DischargeUnitConvert,'QMo-QMi:',(QmatrixOutlet-QmatrixInlet)*DischargeUnitConvert,'Qr:',Qrain*DischargeUnitConvert,'Hmax:',HmatrixMax,'Hmin:',HmatrixMin

            write(lunFScreen,'(6(3xa,f12.8))') 'QMo:',QmatrixOutlet*DischargeUnitConvert,'QMi:',QmatrixInlet*DischargeUnitConvert,'QMo-QMi:',(QmatrixOutlet-QmatrixInlet)*DischargeUnitConvert,'Qr:',Qrain*DischargeUnitConvert,'Hmax:',HmatrixMax,'Hmin:',HmatrixMin
            
            
        END SELECT
    
    end subroutine
    
    subroutine WriteResults(sc)
    
        character(*) sc
        character(len=60) filename,zonename,position
        integer (kind=4) ii,jj,kk,ll,it
        real (kind=8) xp,yp,zp,lp,hp,hpOLD,dhdx,dhdy,dhdz
        real (kind=8) pp,Ks,Ku,vx,vy,vz,Se,ic,SeModes(2),CM3
        
        
        SELECT CASE(sc)
            
        CASE('InitialConditions')
            
            !IC_Matrix approx.
            open(lunF1,file='IC_Matrix_Approx.dat')
            write(lunF1,'(a)') 'TITLE = FFVT_IC_Matrix "'
            write(lunF1,'(a)')  'VARIABLES = "X", "Y", "Z", "IC", "H", "Se" '
            write(lunF1,'(4(2xa,i4))') 'ZONE I=',nz,', J=',ny,', K=',nx,'F=POINT'
               
            do ii=0,nx-1
                do jj=0,ny-1
                    do kk=0,nz-1
                        xp=xVertex(ii)
                        yp=yVertex(jj)
                        zp=zVertex(kk)
                        ic=IC_Matrix(xp,yp,zp)
                        hp=CXYZ(xp,yp,zp,0,0,0,CC_t0)
                        Se=EffectiveSaturation(SoilType(ii,jj,kk),hp-zp,SeModes)
                        write(lunF1,'(6e20.10)') xp,yp,zp,ic,hp,Se
                    enddo
                enddo
            enddo 
            close(lunF1)
            
        CASE('PrepareFile')
            !Matrix solution.
            open(lunF1,file='MATRIX_Results.dat')
            write(lunF1,'(a)') 'TITLE = FFVM_KFM_IC_4.1"'
            write(lunF1,'(a)')  'VARIABLES = "X", "Y", "Z", "Se", "P", "H", "Vx", "Vy", "Vz" '
            close(lunF1)
            
            !Piezometers pressure solution.
            open(lunF1,file='PIEZOMETERS_Head.dat')
            write(lunF1,'(a)') 'TITLE = FFVM_KFM_IC_4.1"'
            write(lunF1,'(a)')  'VARIABLES = "t","A1","A2","A3","A4","A5","A6","A7","A8","A9","A10","A11","B1","B2","B3","B4","B5","B6","B7","B8","B9","B10","B11","C1","C2","C3","C4","C5","C6","C7","C8","C9","C10","C11","D1","D2","D3","D4","D5","D6","D7","D8","D9","D10","D11"'
            close(lunF1)
            
            !!Steady piezometers pressure solution.
            !open(lunF1,file='STEADY_Piezometers_Head.dat')
            !write(lunF1,'(a)') 'TITLE = FFVM_KFM_IC_4.1"'
            !write(lunF1,'(a)')  'VARIABLES = "P","H"'
            !close(lunF1)
            
            open(lunF1,file='PIEZOMETERS_Concentration.dat')
            write(lunF1,'(a)') 'TITLE = FFVM_KFM_IC_4.1"'
            write(lunF1,'(a)')  'VARIABLES = "t","C1","C2","C3","C4","C5","C6","C7","C8","C9","C10","C11","C12","C13","C14","C15","C16","C17","C18","C19","C20"'
            close(lunF1)
            
            !Read piezometers coordinates
            call PiezometerCoordinates
        
        CASE('Solution')
            
            if(mod(icntTimeStep,nWriteResults).ne.0) then
                if(icntTimeStep.ne.1) return
            endif
            
            !Positions where results are written center/edge
            position='center'   !Missing edge
            
            !Matrix solution.
            open(lunF2,file='MATRIX_Results.dat',status='old',position='append')
            write(zonename,'(e14.8)') (TimeCurr)*TimeUnitConvert
            zonename=trim(zonename)
            !write(lunF2,'(a,f12.4,a)')  'TEXT X=70, Y=90, H=5, T="',(TimeCurr)*TimeUnitConvert,'"'
            write(lunF2,'(a,a)') 'ZONE T=t_',trim(zonename)
            if(position.eq.'center') write(lunF2,'(4(2xa,i4))') 'I=',nz+2,', J=',ny+2,', K=',nx+2,'F=POINT' !For results in CV centers
            !if(position.eq.'edge') write(lunF2,'(4(2xa,i4))') 'I=',3*(nz+2)+3,', J=',3*(ny+2)+3,', K=',3*(nx+2)+3,'F=POINT'  !For results in CV edges
            
            HmatrixMin=+1.d+9
            HmatrixMax=-1.d+9
            
            do ii=-1,nx
                do jj=-1,ny
                    do kk=-1,nz
                        !xp=xCenter(ii)
                        !yp=yCenter(jj)
                        !zp=zCenter(kk)
                        xp=xVertex(ii)
                        yp=yVertex(jj)
                        zp=zVertex(kk)
                        if(xp.lt.dlx1) xp=dlx1
                        if(yp.lt.dly1) yp=dly1
                        if(zp.lt.dlz1) zp=dlz1
                        if(xp.gt.dlx2) xp=dlx2
                        if(yp.gt.dly2) yp=dly2
                        if(zp.gt.dlz2) zp=dlz2
                        hp=CXYZ(xp,yp,zp,0,0,0,CC_t1)   
                        !dHdx=CXYZ(xp,yp,zp,1,0,0,CC_t1)
                        !dHdy=CXYZ(xp,yp,zp,0,1,0,CC_t1)
                        !dHdz=CXYZ(xp,yp,zp,0,0,1,CC_t1)
                        pp=hp-zp
                        it=SoilType(min(ii,nx),min(jj,ny),min(kk,nz))
                        Ks=SatCond(it)
                        Se=EffectiveSaturation(it,pp,SeModes)
                        Ku=Ks*RelativePermeability(it,Se,SeModes)
                        vx=-Ku*CXYZ(xp,yp,zp,1,0,0,CC_t1)
                        vy=-Ku*CXYZ(xp,yp,zp,0,1,0,CC_t1)
                        vz=-Ku*CXYZ(xp,yp,zp,0,0,1,CC_t1)/AnisotropZ(it)
                        CM3 = Conc_3D(xp,yp,zp) 
                        !if(icntTimeStep.eq.0) Se=Ks !!!***!!!   !Write saturated conductivity field instead of saturation for initial condition results
                        if(WriteInitCond) Se=Ks
                        write(lunF2,'(14e20.10)') xp,yp,zp,Se,pp,hp,vx,vy,vz,CM3
                        !write(lunF2,'(14e20.10)') xp,yp,zp,Ks,Ku,Se,pp,hp,dhdx,dhdy,dhdz,vx,vy,vz
                        !Remeber h_min value for screen record
                        HmatrixMin=min(HmatrixMin,hp)
                        HmatrixMax=max(HmatrixMax,hp)
                    enddo
                enddo
            enddo 
            close(lunF2)

            !do ii=-1,nx
            !    do jj=-1,ny
            !        do kk=-1,nz
            !            xp=xVertex(ii)  !xCenter(ii)
            !            yp=yVertex(jj)  !yCenter(jj)
            !            zp=zVertex(kk)  !zCenter(kk)
            !            if(xp.lt.dlx1) xp=dlx1
            !            if(yp.lt.dly1) yp=dly1
            !            if(zp.lt.dlz1) zp=dlz1
            !            if(xp.gt.dlx2) xp=dlx2
            !            if(yp.gt.dly2) yp=dly2
            !            if(zp.gt.dlz2) zp=dlz2
            !            !hpOLD=CXYZ(xp,yp,zp,0,0,0,CC_t0)
            !            hp=CXYZ(xp,yp,zp,0,0,0,CC_t1)   
            !            !dHdx=CXYZ(xp,yp,zp,1,0,0,CC_t1)
            !            !dHdy=CXYZ(xp,yp,zp,0,1,0,CC_t1)
            !            !dHdz=CXYZ(xp,yp,zp,0,0,1,CC_t1)
            !            pp=hp-zp
            !            it=SoilType(ii,jj,kk)
            !            Ks=SatCond(it)
            !            Se=EffectiveSaturation(it,pp)
            !            Ku=Ks*RelativePermeability(it,Se)
            !            vx=-Ku*CXYZ(xp,yp,zp,1,0,0,CC_t1)
            !            vy=-Ku*CXYZ(xp,yp,zp,0,1,0,CC_t1)
            !            vz=-Ku*CXYZ(xp,yp,zp,0,0,1,CC_t1)
            !            !VX=MatrixVelocity(xp,yp,zp,1,0,0,CC_LastIter)
            !            !write(lunF1,'(14e20.10)') xp,yp,zp,Ks,Ku,Se,pp,hp,dhdx,dhdy,dhdz,vx,vy,vz
            !            if(icntTimeStep.eq.0) Se=Ks !!!***!!!   !Write saturated conductivity field instead of saturation for initial condition results
            !            write(lunF2,'(14e20.10)') xp,yp,zp,Se,pp,hp,vx,vy,vz
            !        enddo
            !    enddo
            !enddo 
            !close(lunF2)
            
        CASE('VauclinExperimentalData')
            !Matrix solution.
            open(lunF2,file='MATRIX_Results.dat',status='old',position='append')
            write(lunF2,'(a,a)') 'ZONE T=EXPERIMENTAL_t_2h'
            write(lunF2,'(14e20.10)') 0.2788000000E+00,7.5000000000E-01,7.6990000000E-01,0.0000000000E+00,7.6990000000E-01,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00
            write(lunF2,'(14e20.10)') 0.9924000000E+00,7.5000000000E-01,6.9805000000E-01,0.0000000000E+00,6.9805000000E-01,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00
            write(lunF2,'(a,a)') 'ZONE T=EXPERIMENTAL_t_3h'
            write(lunF2,'(14e20.10)') 0.0179000000E+00,7.5000000000E-01,1.0069000000E+00,0.0000000000E+00,1.0069000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00
            write(lunF2,'(14e20.10)') 0.6950000000E+00,7.5000000000E-01,9.0580000000E-01,0.0000000000E+00,9.0580000000E-01,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00
            write(lunF2,'(14e20.10)') 1.6124000000E+00,7.5000000000E-01,7.3520000000E-01,0.0000000000E+00,7.3520000000E-01,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00 
            write(lunF2,'(a,a)') 'ZONE T=EXPERIMENTAL_t_4h'
            write(lunF2,'(14e20.10)') 0.0183000000E+00,7.5000000000E-01,1.1073700000E+00,0.0000000000E+00,1.1073700000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00
            write(lunF2,'(14e20.10)') 0.6936000000E+00,7.5000000000E-01,1.0066000000E+00,0.0000000000E+00,1.0066000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00
            write(lunF2,'(14e20.10)') 1.2922000000E+00,7.5000000000E-01,8.6900000000E-01,0.0000000000E+00,8.6900000000E-01,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00
            write(lunF2,'(14e20.10)') 1.6114000000E+00,7.5000000000E-01,8.0130000000E-01,0.0000000000E+00,8.0130000000E-01,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00
            write(lunF2,'(a,a)') 'ZONE T=EXPERIMENTAL_t_8h'
            write(lunF2,'(14e20.10)') 0.0200000000E+00,7.5000000000E-01,1.1994000000E+00,0.0000000000E+00,1.1994000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00
            write(lunF2,'(14e20.10)') 0.1026000000E+00,7.5000000000E-01,1.1852000000E+00,0.0000000000E+00,1.1852000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00
            write(lunF2,'(14e20.10)') 0.3382000000E+00,7.5000000000E-01,1.1626000000E+00,0.0000000000E+00,1.1626000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00
            write(lunF2,'(14e20.10)') 0.6936000000E+00,7.5000000000E-01,1.1108000000E+00,0.0000000000E+00,1.1108000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00
            write(lunF2,'(14e20.10)') 1.2946000000E+00,7.5000000000E-01,9.8230000000E-01,0.0000000000E+00,9.8230000000E-01,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00
            write(lunF2,'(14e20.10)') 1.6104000000E+00,7.5000000000E-01,9.0480000000E-01,0.0000000000E+00,9.0480000000E-01,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00,0.0000000000E+00
            close(lunF2) 
            
        CASE('PiezometersHead')
            
            !Piezometric head is 44 lower piezometers in cm.
            open(lunF1,file='PIEZOMETERS_Head.dat',status='old',position='append')
            write(lunF1,'(e20.10,44e14.7)') (TimeCurr)*TimeUnitConvert,(CXYZ(LPCoordinates(ii,1),LPCoordinates(ii,2),LPCoordinates(ii,3),0,0,0,CC_t1), ii=1,44)
            close(lunF1)
            
!            !Pressure head is in cm.
!            open(lunF1,file='PACKERS_Head.dat',status='old',position='append')
!            write(lunF1,'(e20.10,44e14.7)') (TimeCurr)*TimeUnitConvert,((CXYZ(UPCoordinates(ii,1),UPCoordinates(ii,2),UPCoordinates(ii,3),0,0,0,CC_t1)), ii=1,4)
!            close(lunF1)
            
        CASE('PiezometersConcentration')
            
            !Piezometric concentration in 20 upper piezometers is in mg/l.
            open(lunF1,file='PIEZOMETERS_Concentration.dat',status='old',position='append')
            write(lunF1,'(e20.10,20e14.7)') (TimeCurr)*TimeUnitConvert,(CXYZ(UPCoordinates(ii,1),UPCoordinates(ii,2),UPCoordinates(ii,3),0,0,0,CCC_t1), ii=1,20)
            close(lunF1)
            
        CASE('SteadyPiezometersHead')
            
            !Piezometric head is in cm.
            open(lunF1,file='STEADY_Piezometers_Head.dat')
            write(lunF1,'(a)') 'TITLE = FFVM_KFM_IC_4.1"'
            write(lunF1,'(a)')  'VARIABLES = "P","A01","A02","A03","A04","A05","A06","A07","A08","A09","A10","A11"'
            do ii=1,44
                if(ii.eq.31.OR.ii.eq.32.OR.ii.eq.42) cycle
                write(lunF1,'(i4,11e14.7)') ii,CXYZ(LPCoordinates(ii,1),LPCoordinates(ii,2),LPCoordinates(ii,3),0,0,0,CC_t1),0.d0,0.d0,0.d0,0.d0,0.d0,0.d0,0.d0,0.d0,0.d0,0.d0
            enddo
            close(lunF1)
            
        END SELECT
        
    end subroutine
    
         Real*8 function CourantMatrixTimeStep

             integer*4   ii,jj,kk
             real*8      vx
             
             CourantMatrixTimeStep =1.d9
             
             do ii=-1,nx
                do jj=-1,ny
                    do kk=-1,nz
                        !xp=xCenter(ii)
                        !yp=yCenter(jj)
                        !zp=zCenter(kk)
                        xp=xVertex(ii)
                        yp=yVertex(jj)
                        zp=zVertex(kk)
                        if(xp.lt.dlx1) xp=dlx1
                        if(yp.lt.dly1) yp=dly1
                        if(zp.lt.dlz1) zp=dlz1
                        if(xp.gt.dlx2) xp=dlx2
                        if(yp.gt.dly2) yp=dly2
                        if(zp.gt.dlz2) zp=dlz2
                    
                     vx = dabs(MatrixQVelocity(xp,yp,zp,1,0,0,CC_Lastiter))
                     vy = dabs(MatrixQVelocity(xp,yp,zp,0,1,0,CC_Lastiter))
                     vz = dabs(MatrixQVelocity(xp,yp,zp,0,0,1,CC_Lastiter))
                     
                     if (dabs(vx).gt.1.d-12) CourantMatrixTimeStep = dmin1(dx/(2.0d0*v_x),CourantMatrixTimeStep)
                     if (dabs(vy).gt.1.d-12) CourantMatrixTimeStep = dmin1(dy/(2.0d0*v_y),CourantMatrixTimeStep)
                     if (dabs(vz).gt.1.d-12) CourantMatrixTimeStep = dmin1(dz/(2.0d0*v_z),CourantMatrixTimeStep)
                    
                    
                    enddo
                enddo
            enddo 
            
            
       end function CourantMatrixTimeStep   
        
    subroutine CalculateMatrixDischarge 
    
        integer (kind=4) ii,jj,kk,it
        real (kind=8) CVdx,CVdy,CVdz,xp,yp,zp
        real (kind=8) Ks,kr,Se_c,hC,dhCdn,vx,vz,Integ,SeModes_C(2)
        !real (kind=8) QmatrixInlet,QmatrixOutlet
        
        !Set to zero
        QmatrixInlet=0.d0
        QmatrixOutlet=0.d0
        !
        !Matrix discharge through upstream and downstream reservoir plane
        do ii=-1,nx,nx+1
            if(ii.eq.-nExternal) xp=dlx1
            if(ii.eq.nx)         xp=dlx2
            do jj=-1,ny
                yp=yCenter(jj)
                CVdy=ycp2(jj)-ycp1(jj)
                do kk=-1,nz
                    zp=zCenter(kk)
                    CVdz=zcp2(kk)-zcp1(kk)
                    hC=CXYZ(xp,yp,zp,0,0,0,CC_t1)
                    it=SoilType(ii,jj,kk)
                    Se_C=EffectiveSaturation(it,hC-zp,SeModes_C)
                    kr=RelativePermeability(it,Se_C,SeModes_C)
                    if(kr.lt.1.d0) cycle
                    Ks=SatCond(it)
                    dhCdn=CXYZ(xp,yp,zp,1,0,0,CC_t1)
                    vx=-kr*Ks*dhCdn
                    Integ=vx*CVdy*CVdz
                    !call trap_int_surface_integ_3Dfun_MatrixVelocities(MatrixVelocity,CC_t1,1,0,0,xp,xp,ycp1(jj),ycp2(jj),zcp1(kk),zcp2(kk),kmaxMV,Integ)
                    if(ii.eq.-nExternal) QmatrixInlet=QmatrixInlet+Integ
                    if(ii.eq.nx)         QmatrixOutlet=QmatrixOutlet+Integ
                enddo
            enddo
        enddo
        
        
        !!Matrix discharge through top and bottom reservoir plane
        !do kk=-1,nz,nz+1
        !    if(kk.eq.-nExternal) zp=dlz1
        !    if(kk.eq.nz)         zp=dlz2
        !    do jj=-1,ny
        !        yp=yCenter(jj)
        !        CVdy=ycp2(jj)-ycp1(jj)
        !        do ii=-1,nx
        !            xp=xCenter(ii)
        !            CVdx=xcp2(ii)-xcp1(ii)
        !            hC=CXYZ(xp,yp,zp,0,0,0,CC_t1)
        !            it=SoilType(ii,jj,kk)
        !            Se_C=EffectiveSaturation(it,hC-zp,SeModes_C)
        !            kr=RelativePermeability(it,Se_C,SeModes_C)
        !            !if(kr.lt.1.d0) cycle   !!!***!!!
        !            Ks=SatCond(it)
        !            dhCdn=CXYZ(xp,yp,zp,0,0,1,CC_t1)
        !            vz=-kr*Ks*dhCdn
        !            Integ=vz*CVdx*CVdy
        !            if(kk.eq.-nExternal) QmatrixOutlet=QmatrixOutlet+Integ
        !            if(kk.eq.nz)         QmatrixInlet=QmatrixInlet-Integ
        !        enddo
        !    enddo
        !enddo

        

    end subroutine
        
    subroutine PiezometerCoordinates

        !Lower piezometers coordinates
        !A
        LPCoordinates(1,1:3)=(/3.83	,	0.54	,	0.77/)
        LPCoordinates(2,1:3)=(/3.37	,	0.54	,	1.00/)
        LPCoordinates(3,1:3)=(/3.20	,	0.58	,	0.32/)
        LPCoordinates(4,1:3)=(/2.75	,	0.46	,	0.75/)
        LPCoordinates(5,1:3)=(/2.45	,	0.22	,	1.00/)
        LPCoordinates(6,1:3)=(/1.98	,	0.47	,	0.42/)  !(/2.02	,	0.47	,	0.42/)  !A6
        LPCoordinates(7,1:3)=(/1.65	,	0.66	,	0.65/)
        LPCoordinates(8,1:3)=(/1.08	,	0.77	,	1.06/)  !(/1.03	,	0.77	,	1.00/)  !A8
        LPCoordinates(9,1:3)=(/0.97	,	0.60	,	0.33/)
        LPCoordinates(10,1:3)=(/0.78	,	0.10	,	1.00/)
        LPCoordinates(11,1:3)=(/0.65	,	0.45	,	0.70/)
        !B
        LPCoordinates(12,1:3)=(/3.83	,	0.96	,	0.33/)
        LPCoordinates(13,1:3)=(/3.19	,	0.91	,	0.70/)
        LPCoordinates(14,1:3)=(/3.18	,	0.97	,	1.00/)
        LPCoordinates(15,1:3)=(/2.74	,	1.00	,	0.33/)
        LPCoordinates(16,1:3)=(/2.75	,	0.86	,	0.65/)
        LPCoordinates(17,1:3)=(/2.22	,	1.20	,	1.00/)
        LPCoordinates(18,1:3)=(/1.86	,	1.10	,	0.43/)  !(/1.64	,	1.10	,	0.33/)  !B7
        LPCoordinates(19,1:3)=(/1.51	,	1.36	,	0.60/)  !(/1.40	,	1.23	,	0.60/)  !B8
        LPCoordinates(20,1:3)=(/0.75	,	1.24	,	1.00/)
        LPCoordinates(21,1:3)=(/0.53	,	1.11	,	0.33/)
        LPCoordinates(22,1:3)=(/0.05	,	1.02	,	0.60/)
        !C
        LPCoordinates(23,1:3)=(/3.53	,	1.44	,	1.06/)
        LPCoordinates(24,1:3)=(/3.48	,	1.55	,	0.33/)
        LPCoordinates(25,1:3)=(/2.79	,	1.55	,	0.65/)  !(/2.89	,	1.47	,	0.55/)  !C3
        LPCoordinates(26,1:3)=(/2.85	,	1.80	,	0.95/)
        LPCoordinates(27,1:3)=(/2.34	,	1.64	,	0.39/)  !(/2.38	,	1.64	,	0.34/)  !C4
        LPCoordinates(28,1:3)=(/2.12	,	1.68	,	0.65/)  !(/2.12	,	1.65	,	0.70/)  !C6
        LPCoordinates(29,1:3)=(/1.97	,	1.73	,	1.02/)  !(/1.87	,	1.73	,	1.00/)  !C7
        LPCoordinates(30,1:3)=(/1.31	,	1.54	,	0.31/)  !(/1.24	,	1.54	,	0.31/)  !C8
        LPCoordinates(31,1:3)=(/0.82	,	1.62	,	0.60/)
        LPCoordinates(32,1:3)=(/0.88	,	1.56	,	1.00/)
        LPCoordinates(33,1:3)=(/0.14	,	1.76	,	0.33/)
        !D
        LPCoordinates(34,1:3)=(/3.89	,	2.07	,	0.33/)
        LPCoordinates(35,1:3)=(/3.23	,	2.27	,	0.70/)
        LPCoordinates(36,1:3)=(/2.90	,	2.26	,	1.00/)
        LPCoordinates(37,1:3)=(/2.83	,	2.10	,	0.34/)
        LPCoordinates(38,1:3)=(/2.20	,	1.98	,	0.69/)  !(/2.20	,	1.94	,	0.65/)  !D5
        LPCoordinates(39,1:3)=(/2.12	,	1.96	,	1.20/)
        LPCoordinates(40,1:3)=(/1.86	,	2.13	,	0.36/)  !(/1.69	,	2.13	,	0.34/)  !D7
        LPCoordinates(41,1:3)=(/1.16	,	2.40	,	0.60/)  !(/1.03	,	2.40	,	0.60/)  !D8
        LPCoordinates(42,1:3)=(/0.70	,	2.02	,	1.00/)
        LPCoordinates(43,1:3)=(/0.57	,	2.24	,	0.34/)
        LPCoordinates(44,1:3)=(/0.35	,	2.25	,	0.85/)  !(/0.30	,	2.25	,	0.85/)
        
        !!Upper piezometers coordinates
        !UPCoordinates(1,1:3)=(/3.24	,	1.84	,	0.58/)  !(/3.24	,	1.84	,	0.58/)
        !UPCoordinates(2,1:3)=(/2.00	,	1.74	,	0.81/)  !(/2.00	,	1.74	,	0.81/)
        !UPCoordinates(3,1:3)=(/1.01	,	1.44	,	0.76/)  !(/1.01	,	1.53	,	0.76/)
        !UPCoordinates(4,1:3)=(/0.48	,	1.54	,	0.65/)  !(/0.48	,	1.54	,	0.65/)
        
        !Upper piezometers coordinates
        UPCoordinates(1,1:3) =(/3.52	,	1.54	,	1.10/)  !(/3.22	,	1.45	,	0.583/)  !(/3.24	,	1.84	,	0.58/)
        UPCoordinates(2,1:3) =(/3.45	,   1.24	,   1.00/)  !(/2.00	,	1.71	,	0.815/)  !(/2.00	,	1.74	,	0.81/)
        UPCoordinates(3,1:3) =(/3.18	,	1.66	,	0.90/)  !(/1.02	,	1.63	,	0.750/)  !(/1.01	,	1.53	,	0.76/)
        UPCoordinates(4,1:3) =(/3.14	,	0.85	,	0.90/)  !(/0.48	,	1.51	,	0.638/)  !(/0.48	,	1.54	,	0.65/)
        UPCoordinates(5,1:3) =(/2.98	,	0.97	,	1.00/)  !(/3.22	,	1.45	,	0.583/)  !(/3.24	,	1.84	,	0.58/)
        UPCoordinates(6,1:3) =(/2.96	,	0.58	,	0.80/)  !(/2.00	,	1.71	,	0.815/)  !(/2.00	,	1.74	,	0.81/)
        UPCoordinates(7,1:3) =(/2.30	,	1.74	,	0.90/)  !(/1.02	,	1.63	,	0.750/)  !(/1.01	,	1.53	,	0.76/)
        UPCoordinates(8,1:3) =(/2.36	,	1.10	,	0.80/)  !(/0.48	,	1.51	,	0.638/)  !(/0.48	,	1.54	,	0.65/)
        UPCoordinates(9,1:3) =(/2.41	,	0.76	,	0.70/)  !(/3.22	,	1.45	,	0.583/)  !(/3.24	,	1.84	,	0.58/)
        UPCoordinates(10,1:3)=(/2.04	,	1.72	,	0.90/)  !(/2.00	,	1.71	,	0.815/)  !(/2.00	,	1.74	,	0.81/)
        UPCoordinates(11,1:3)=(/1.96	,	0.87	,	0.90/)  !(/1.02	,	1.63	,	0.750/)  !(/1.01	,	1.53	,	0.76/)
        UPCoordinates(12,1:3)=(/1.50	,   0.58	,   0.95/)  !(/0.48	,	1.51	,	0.638/)  !(/0.48	,	1.54	,	0.65/)
        UPCoordinates(13,1:3)=(/1.03	,   1.77	,   0.90/)  !(/3.22	,	1.45	,	0.583/)  !(/3.24	,	1.84	,	0.58/)
        UPCoordinates(14,1:3)=(/1.05    ,	0.90	,   1.00/)  !(/2.00	,	1.71	,	0.815/)  !(/2.00	,	1.74	,	0.81/)
        UPCoordinates(15,1:3)=(/1.96	,	0.87	,	0.90/)  !(/1.02	,	1.63	,	0.750/)  !(/1.01	,	1.53	,	0.76/)
        UPCoordinates(16,1:3)=(/0.48	,	1.54	,	0.63/)  !(/0.48	,	1.51	,	0.638/)  !(/0.48	,	1.54	,	0.65/)
        UPCoordinates(17,1:3)=(/0.76	,   1,84	,   1.15/)  !(/3.22	,	1.45	,	0.583/)  !(/3.24	,	1.84	,	0.58/)
        UPCoordinates(18,1:3)=(/0.75	,	1.05	,	1.35/)  !(/2.00	,	1.71	,	0.815/)  !(/2.00	,	1.74	,	0.81/)
        UPCoordinates(19,1:3)=(/0.01	,	1.53	,	0.75/)  !(/1.02	,	1.63	,	0.750/)  !(/1.01	,	1.53	,	0.76/)
        UPCoordinates(20,1:3)=(/0.48	,	1.54	,	0.63/)  !(/0.48	,	1.51	,	0.638/)  !(/0.48	,	1.54	,	0.65/)
 
 
        !Coordinate system is in opposite direction on paper and code
        LPCoordinates(1:44,1)=x2Matrix-LPCoordinates(1:44,1)
        UPCoordinates(1:20,1)=x2Matrix-UPCoordinates(1:4,1)
        
        !Height is measured at the middle of isolated part.
        !UPCoordinates(1:4,3)=UPCoordinates(1:4,3)-0.08d0

    end subroutine
     
!___________________________________________________________!
!PRIVATE FUNCTIONS

    real(kind=8) function BFUN3D(n,xv,yv,zv,xp,yp,zp,dx,dy,dz,mdx,mdy,mdz)
    !Returns 3D basis function value as product of 1D basis functions.
    
        integer(kind=4) n,mdx,mdy,mdz
        real(kind=8) xv,yv,zv,xp,yp,zp,dx,dy,dz

        BFUN3D=bfun(n,xv,xp,dx,mdx,dlx1,dlx2) &
              *bfun(n,yv,yp,dy,mdy,dly1,dly2) &
              *bfun(n,zv,zp,dz,mdz,dlz1,dlz2) 

    end function

    real(kind=8) function CXYZ(xp,yp,zp,mdx,mdy,mdz,CC)
    !Returns calculated solution (md? derivative) value in arbitrary point.
    
        integer(kind=4) mdx,mdy,mdz
        integer(kind=4) indx,indy,indz
        integer(kind=4) ix,jy,kz,ic,jc,kc,jcnt
        real(kind=8) xp,yp,zp,CC(*)

        CXYZ=0.d0
        !Basis function vertex position
        ix=int4((xp-dlx1)/(dx))
        jy=int4((yp-dly1)/(dy))
        kz=int4((zp-dlz1)/(dz))
        
        !All non-zero coefficinets
        do ic=iX-1,iX+2
            if(ic.lt.-nExternal.OR.ic.gt.nx-1+nExternal) cycle
            do jc=jY-1,jY+2
                if(jc.lt.-nExternal.OR.jc.gt.ny-1+nExternal) cycle
                do kc=kZ-1,kZ+2
                    if(kc.lt.-nExternal.OR.kc.gt.nz-1+nExternal) cycle
                    !Ordinal number of basis function
                    jcnt=(kc+nExternal+1)+(jc-1+nExternal+1)*nzTOT+(ic-1+nExternal+1)*nzTOT*nyTOT
                    !Sum all basis function values multiplied by corresponding coefficients                
                    CXYZ=CXYZ+CC(jcnt)*BFUN3D(norder,xVertex(ic),yVertex(jc),zVertex(kc),xp,yp,zp,dx,dy,dz,mdx,mdy,mdz)             
                enddo
            enddo
        enddo

    end function


    real(kind=8) function Disp_ij(ip,jp,xp,yp,zp)
    !Returns calculated component value ip,jp of dispersion tensor in point (xp,yp,zp).
    
        integer(kind=4) ip,jp
        real(kind=8) xp,yp,zp
        real(kind=8) vx,vy,vz,vel

        !Calculate valocity in point (xp,yp,zp)
        vx = Vm3D(xp,yp,zp,1,0,0)
        vy = Vm3D(xp,yp,zp,0,1,0)
        vz = Vm3D(xp,yp,zp,0,0,1)
        vel = dsqrt(vx**2+vy**2+vz**2)
        
        !Point indices
        if (ip.eq.1) v_ip = vx
        if (ip.eq.2) v_ip = vy        
        if (ip.eq.3) v_ip = vz       
        if (jp.eq.1) v_jp = vx
        if (jp.eq.2) v_jp = vy        
        if (jp.eq.3) v_jp = vz       

        Disp_ij = 0.0d0
        if (ip.eq.jp) Disp_ij = Disp_ij + alfa_t*v_ip
        Disp_ij = Disp_ij + (alfa_l-alfa_t)*v_ip*v_jp/vel


    end function

     real(kind=8) function MatrixVelocity(xp,yp,zp,mdx,mdy,mdz,CC)
    !Returns calculated matrix Darcy velocity value in point (xp,yp,zp).
    
        integer(kind=4) mdx,mdy,mdz
        integer(kind=4) ix,jy,kz,it
        real(kind=8) xp,yp,zp,CC(*)
        real(kind=8) Ks,hm,dhmdn,Se,SeModes(2)

        !Point indexes
        ix=int4((xp-dlx1)/(dx))
        jy=int4((yp-dly1)/(dy))
        kz=int4((zp-dlz1)/(dz))
        !Saturated conductivity
        it=SoilType(ix,jy,kz)
        Ks=SatCond(it)
        if(mdz.eq.1) Ks=Ks/AnisotropZ(it)
        
        !Head value
        hm=CXYZ(xp,yp,zp,0,0,0,CC)
        !Directional derivative value (in x, y or z direction)
        dhmdn=CXYZ(xp,yp,zp,mdx,mdy,mdz,CC)
        !Saturation value
        Se=EffectiveSaturation(it,hm-zp,SeModes)
!        O = WaterContent(it,Se)
        !Velocity value
        MatrixVelocity=-Ks*RelativePermeability(it,Se,SeModes)*dhmdn

    end function


     real(kind=8) function MatrixQVelocity(xp,yp,zp,mdx,mdy,mdz,CC)
    !Returns real calculated matrix velocity value in point (xp,yp,zp).
    !Darcy velocity divided by water content
    
        integer(kind=4) mdx,mdy,mdz
        integer(kind=4) ix,jy,kz,it
        real(kind=8) xp,yp,zp,CC(*),O
        real(kind=8) Ks,hm,dhmdn,Se,SeModes(2)

        !Point indexes
        ix=int4((xp-dlx1)/(dx))
        jy=int4((yp-dly1)/(dy))
        kz=int4((zp-dlz1)/(dz))
        !Saturated conductivity
        it=SoilType(ix,jy,kz)
        Ks=SatCond(it)
        if(mdz.eq.1) Ks=Ks/AnisotropZ(it)
        
        !Head value
        hm=CXYZ(xp,yp,zp,0,0,0,CC)
        !Directional derivative value (in x, y or z direction)
        dhmdn=CXYZ(xp,yp,zp,mdx,mdy,mdz,CC)
        !Saturation value
        Se=EffectiveSaturation(it,hm-zp,SeModes)
        O = WaterContent(it,Se)
        !Velocity value
        MatrixQVelocity = -Ks*RelativePermeability(it,Se,SeModes)*dhmdn
        MatrixQVelocity = MatrixQVelocity/O
    end function

    real(kind=8) function SijFUN(it,Se,xv,yv,zv,xp,yp,zp)
    !Returnes function value for numerical integration of part of mass-matrix coefficients.  
    
        integer(kind=4) it
        real(kind=8) Se,xv,yv,zv,xp,yp,zp

        !S=S(Se)*BFUN
        SijFUN=Saturation(it,Se)*BFUN3D(nOrder,xv,yv,zv,xp,yp,zp,dx,dy,dz,0,0,0)
                
    end function

    real(kind=8) function MLSijFUN(it,Se,xv,yv,zv,xp,yp,zp)
    !Returnes mass-lumped function value for numerical integration of part of mass-matrix coefficients.  
    
        integer(kind=4) it
        real(kind=8) Se,xv,yv,zv,xp,yp,zp

        !S=S(Se)
        MLSijFUN=Saturation(it,Se)
                
    end function

    real(kind=8) function CijFUN(it,Se,SeModes,xv,yv,zv,xp,yp,zp)
    !Returnes function value for numerical integration of part of mass-matrix coefficients.  
    
        integer(kind=4) it
        real(kind=8) Se,SeModes(2),xv,yv,zv,xp,yp,zp

        !C=C(Se)*BFUN
        CijFUN=SpecMoistCapacity(it,Se,SeModes)*BFUN3D(nOrder,xv,yv,zv,xp,yp,zp,dx,dy,dz,0,0,0)
                
    end function

    real(kind=8) function MLCijFUN(it,Se,SeModes,xv,yv,zv,xp,yp,zp)
    !Returnes mass-lumped function value for numerical integration of part of mass-matrix coefficients.  
    
        integer(kind=4) it
        real(kind=8) Se,SeModes(2),xv,yv,zv,xp,yp,zp

        !C=C(Se)
        MLCijFUN=SpecMoistCapacity(it,Se,SeModes)
                
    end function

    real(kind=8) function KijFUN(it,Se,SeModes,xv,yv,zv,xp,yp,zp,mdx,mdy,mdz)
    !Returnes function value for numerical integration of conductivity-matrix coefficients.  
    
        integer(kind=4) it,mdx,mdy,mdz
        real(kind=8) Se,SeModes(2),xv,yv,zv,xp,yp,zp

        !K=(kr(Se))*grad(BFUN)*n
        KijFUN=RelativePermeability(it,Se,SeModes)*BFUN3D(nOrder,xv,yv,zv,xp,yp,zp,dx,dy,dz,mdx,mdy,mdz)
                
    end function            

    real(kind=8) function OOWijFUN(it,Se,xv,yv,zv,xp,yp,zp)
    !Returnes function value for numerical integration of RHS coefficient.  
    
        integer(kind=4) it
        real(kind=8) Se,xv,yv,zv,xp,yp,zp
        real(kind=8) hp_t,Se_t,SeModes(2)  !Solution in time "t"

        !OOW=O^(*)+O^(t)+dt*W^(*)
        hp_t=CXYZ(xp,yp,zp,0,0,0,CC_t0)
        Se_t=EffectiveSaturation(it,hp_t-zp,SeModes)
        OOWijFUN=cfSS3D*Porosity(it)*(-Saturation(it,Se)+Saturation(it,Se_t))+dt3D*SourceTerm(xp,yp,zp)
                
    end function    
!___________________________________________________________!

END MODULE
    
